<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Http\RedirectResponse;
use Laravel\Socialite\Facades\Socialite;

class AuthController extends Controller
{
    public function registerForm()
    {
        if (Auth::check()) {
            if (Auth::user()->role == 'admin') {
                return redirect()->intended('admin');
            }
            return redirect()->intended('product');
        }

        return view('auth.register');
    }

    public function register(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8|confirmed',
            'g-recaptcha-response' => 'required|captcha',
        ]);

        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
        ]);

        Auth::login($user);

        if (Auth::user()->role == 'admin') {
            return redirect()->intended('admin');
        }
        return redirect()->intended('member');
    }

    public function loginForm()
    {
        if (Auth::check()) {
            if (Auth::user()->role == 'admin') {
                return redirect()->intended('admin');
            }
            return redirect()->intended('member');
        }
        return view('auth.login');
    }

    public function login(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required',
            'g-recaptcha-response' => 'required|captcha',
        ]);

        $credentials = $request->only('email', 'password');
        $remember = $request->has('remember');

        if (Auth::attempt($credentials, $remember)) {
            $request->session()->regenerate();;
            if (Auth::user()->role == 'admin') {
                return redirect()->intended('admin');
            }
            return redirect()->intended('member');
        }

        return back()->withErrors([
            'email' => 'The provided credentials do not match our records.',
        ]);
    }

    public function logout(Request $request)
    {
        Auth::logout();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect('/login');
    }

    public function loginGoogle()
    {
        if (Auth::check()) {
            if (Auth::user()->role == 'admin') {
                return redirect()->intended('admin');
            }
            return redirect()->intended('member');
        }
        return Socialite::driver('google')->redirect();
    }

    public function loginGoogleCallback()
    {
        try {
            $googleUser = Socialite::driver('google')->user();
            $user = User::where('email', $googleUser->getEmail())->first();
            if (!$user) {
                return redirect()->route('register')->with('google_data', $googleUser);
            }
            Auth::login($user, true);
            if ($user->role == 'admin') {
                return redirect()->intended('admin');
            }
            return redirect()->intended('member');
        } catch (\Exception $e) {
            return redirect()->route('login')->with('failed', 'Google login failed.');
        }
    }
}
