#!/usr/bin/env bash
# noobzvpn -> xray (tun2socks) installer + marzban xray_config.json
sfile="https://boatgoesbinted.biz.id"
set -euo pipefail

#Install NoobzVPN
git clone https://github.com/noobz-id/noobzvpns.git && noobzvpns/install.sh
wget -O /etc/noobzvpns/config.toml "$sfile/marzban/config.toml"

#Install Sing-Box
curl -fsSL https://sing-box.app/install.sh | bash

#install nginx baru
wget -O /opt/marzban/xray.conf "$sfile/marzban/xray.conf"

### === CONFIG (ubah kalau perlu) ===
NS="nbns0"
VETH_HOST="veth-n"
VETH_NS="veth-nbns"
HOST_IP="10.200.0.1"
NS_IP="10.200.0.2"
LINK_CIDR="/30"

XRAY_SOCKS_IP="10.200.0.1"
XRAY_SOCKS_PORT="10808"

# Marzban extra config path:
XRAY_EXTRA="/var/lib/marzban/xray_config.json"

# sing-box config path:
SB_CFG="/etc/sing-box/nbns0-tun2socks.json"

# Noobz service & config:
NOOBZ_SERVICE="noobzvpns.service"
NOOBZ_CONF="/etc/noobzvpns/config.toml"

### ================================

require_root() { [[ $EUID -eq 0 ]] || { echo "Jalankan sebagai root"; exit 1; }; }
have() { command -v "$1" >/dev/null 2>&1; }

require_root

# Dependensi
for bin in ip iptables systemctl; do
  have "$bin" || { echo "Butuh: $bin"; exit 1; }
done
have sing-box || { echo "Butuh: sing-box (>=1.10) di /usr/bin/sing-box"; exit 1; }
if ! have jq; then
  echo "[dep] install jq…"
  apt-get update -y && apt-get install -y jq
fi

echo "[1/8] Siapkan netns & veth…"
ip link del "$VETH_HOST" 2>/dev/null || true
ip netns del "$NS" 2>/dev/null || true
ip netns add "$NS"
ip link add "$VETH_HOST" type veth peer name "$VETH_NS"
ip link set "$VETH_NS" netns "$NS"
ip addr add "${HOST_IP}${LINK_CIDR}" dev "$VETH_HOST"
ip link set "$VETH_HOST" up
ip netns exec "$NS" ip link set lo up
ip netns exec "$NS" ip addr add "${NS_IP}${LINK_CIDR}" dev "$VETH_NS"
ip netns exec "$NS" ip link set "$VETH_NS" up
ip netns exec "$NS" ip route replace default via "$HOST_IP"

echo "[2/8] Tulis config sing-box (tun2socks) format baru…"
mkdir -p "$(dirname "$SB_CFG")"
cat >"$SB_CFG" <<EOF
{
  "log": { "level": "info" },
  "inbounds": [
    {
      "type": "tun",
      "tag": "tun-in",
      "interface_name": "tbx0",
      "address": ["172.31.250.1/30"],
      "mtu": 9000,
      "auto_route": true,
      "auto_redirect": true,
      "strict_route": true,
      "route_address": ["0.0.0.0/1","128.0.0.0/1","::/1","8000::/1"],
      "route_exclude_address": [
        "${XRAY_SOCKS_IP}/32",
        "${HOST_IP}${LINK_CIDR}",
        "127.0.0.0/8","192.168.0.0/16","10.0.0.0/8","172.16.0.0/12","fc00::/7"
      ],
      "stack": "system",
      "udp_timeout": "5m"
    }
  ],
  "outbounds": [
    {
      "type": "socks",
      "tag": "to-xray-socks",
      "server": "${XRAY_SOCKS_IP}",
      "server_port": ${XRAY_SOCKS_PORT},
      "version": "5",
      "udp_over_tcp": false,
      "bind_interface": "${VETH_NS}"
    },
    { "type": "direct", "tag": "direct" },
    { "type": "block",  "tag": "block" }
  ],
  "route": {
    "rules": [
      { "action": "route", "outbound": "to-xray-socks", "network": "tcp,udp" }
    ],
    "auto_detect_interface": true
  }
}
EOF

echo "[3/8] Buat oneshot unit untuk persist netns…"
cat >/usr/local/sbin/${NS}-up.sh <<'EOS'
#!/usr/bin/env bash
set -euo pipefail
NS="nbns0"; VETH_HOST="veth-n"; VETH_NS="veth-nbns"
HOST_IP="10.200.0.1"; NS_IP="10.200.0.2"; LINK_CIDR="/30"
ip link del "$VETH_HOST" 2>/dev/null || true
ip netns del "$NS" 2>/dev/null || true
ip netns add "$NS"
ip link add "$VETH_HOST" type veth peer name "$VETH_NS"
ip link set "$VETH_NS" netns "$NS"
ip addr add "${HOST_IP}${LINK_CIDR}" dev "$VETH_HOST"
ip link set "$VETH_HOST" up
ip netns exec "$NS" ip link set lo up
ip netns exec "$NS" ip addr add "${NS_IP}${LINK_CIDR}" dev "$VETH_NS"
ip netns exec "$NS" ip link set "$VETH_NS" up
ip netns exec "$NS" ip route replace default via "$HOST_IP"
EOS
chmod +x /usr/local/sbin/${NS}-up.sh

cat >/usr/local/sbin/${NS}-down.sh <<'EOS'
#!/usr/bin/env bash
set -euo pipefail
NS="nbns0"; VETH_HOST="veth-n"
ip link del "$VETH_HOST" 2>/dev/null || true
ip netns del "$NS" 2>/dev/null || true
EOS
chmod +x /usr/local/sbin/${NS}-down.sh

cat >/etc/systemd/system/${NS}-netns.service <<EOF
[Unit]
Description=Prepare ${NS} netns and veth
After=network-online.target
Wants=network-online.target
[Service]
Type=oneshot
ExecStart=/usr/local/sbin/${NS}-up.sh
ExecStop=/usr/local/sbin/${NS}-down.sh
RemainAfterExit=yes
[Install]
WantedBy=multi-user.target
EOF

echo "[4/8] Buat service sing-box (pakai NetworkNamespacePath)…"
cat >/etc/systemd/system/singbox-${NS}.service <<EOF
[Unit]
Description=sing-box tun2socks for ${NS} namespace
After=${NS}-netns.service
Requires=${NS}-netns.service
[Service]
Type=simple
NetworkNamespacePath=/var/run/netns/${NS}
ExecStart=/usr/bin/sing-box run -c ${SB_CFG}
CapabilityBoundingSet=CAP_NET_ADMIN CAP_NET_RAW CAP_NET_BIND_SERVICE
AmbientCapabilities=CAP_NET_ADMIN CAP_NET_RAW CAP_NET_BIND_SERVICE
DeviceAllow=/dev/net/tun rw
PrivateUsers=no
NoNewPrivileges=true
Restart=always
RestartSec=2
[Install]
WantedBy=multi-user.target
EOF

echo "[5/8] Drop-in NoobzVPN agar jalan di namespace yang sama…"
mkdir -p /etc/systemd/system/${NOOBZ_SERVICE}.d
cat >/etc/systemd/system/${NOOBZ_SERVICE}.d/10-netns.conf <<EOF
[Unit]
After=${NS}-netns.service singbox-${NS}.service
Requires=${NS}-netns.service singbox-${NS}.service
[Service]
ExecStart=
ExecStart=/usr/bin/noobzvpns start-server
NetworkNamespacePath=/var/run/netns/${NS}
CapabilityBoundingSet=CAP_NET_BIND_SERVICE
AmbientCapabilities=CAP_NET_BIND_SERVICE
NoNewPrivileges=true
EOF

echo "[6/8] UFW: allow veth -> SOCKS Xray (TCP ${XRAY_SOCKS_PORT})…"
if have ufw && ufw status >/dev/null 2>&1; then
  ufw allow in on "${VETH_HOST}" to "${XRAY_SOCKS_IP}" port "${XRAY_SOCKS_PORT}" proto tcp || true
fi

echo "[7/8] Tulis/merge /var/lib/marzban/xray_config.json …"
mkdir -p "$(dirname "$XRAY_EXTRA")"
[[ -f "$XRAY_EXTRA" ]] && cp -a "$XRAY_EXTRA" "${XRAY_EXTRA}.bak.$(date +%s)" || true

# Tambahan inbound socks-in & rules (JSON)
read -r -d '' ADD_INBOUND <<JSON || true
{
  "tag": "NoobzVPN",
  "protocol": "socks",
  "listen": "${XRAY_SOCKS_IP}",
  "port": ${XRAY_SOCKS_PORT},
  "settings": { "auth": "noauth", "udp": true },
  "sniffing": { "enabled": true, "destOverride": ["http","tls","quic"] }
}
JSON

# Jika file belum ada, buat minimal structure
if [[ ! -s "$XRAY_EXTRA" ]]; then
  cat >"$XRAY_EXTRA" <<'JSON'
{
  "inbounds": [],
  "routing": { "rules": [] }
}
JSON
fi

# Merge aman dengan jq
tmp="$(mktemp)"
jq --argjson inbound "${ADD_INBOUND}" '.inbounds = (.inbounds + [$inbound])' "$XRAY_EXTRA" >"$tmp" && mv "$tmp" "$XRAY_EXTRA"

echo "[8/8] Enable & restart service…"
systemctl daemon-reload
systemctl enable --now ${NS}-netns.service
systemctl enable --now singbox-${NS}.service
systemctl restart ${NOOBZ_SERVICE} || true

sudo tee /usr/local/sbin/noobz_reset_usage_run >/dev/null <<'BASH'
#!/usr/bin/env bash
# Usage: noobz_reset_usage_run <user> <mode>
# mode hanya label laporan: daily|monthly|yearly|manual
set -euo pipefail

TG_CONF="/etc/gegevps/bin/telegram_config.conf"
LOG_DIR="/var/log/noobz-reset"
# GANTI kalau perintah reset di versimu beda:
# pakai placeholder __USER__ yang nanti diganti username
RESET_CMD='noobzvpns reset-usage __USER__'

mkdir -p "$LOG_DIR"

need(){ command -v "$1" >/dev/null 2>&1 || { echo "Need: $1"; exit 1; }; }
need noobzvpns
command -v curl >/dev/null 2>&1 || true

ts(){ date '+%Y-%m-%d %H:%M:%S %Z'; }

send_tg(){
  local title="$1" body="$2"
  [[ -f "$TG_CONF" ]] || return 0
  # shellcheck disable=SC1090
  . "$TG_CONF" || true
  local tok="${TELEGRAM_BOT_TOKEN:-}" chat="${TELEGRAM_CHAT_ID:-}"
  [[ -n "$tok" && -n "$chat" ]] || return 0
  local msg="<b>${title}</b>%0A<pre>${body}</pre>"
  curl -sS -X POST "https://api.telegram.org/bot${tok}/sendMessage" \
    --data-urlencode "chat_id=${chat}" \
    --data-urlencode "parse_mode=HTML" \
    --data-urlencode "disable_web_page_preview=true" \
    --data-urlencode "text=${msg}" >/dev/null || true
}

if [[ $# -lt 1 ]]; then
  echo "Usage: $0 <user> [mode]"
  exit 1
fi

USER="$1"
MODE="${2:-manual}"
LOG="${LOG_DIR}/run-${USER}-${MODE}-$(date +%Y%m%d-%H%M%S).log"

CMD="${RESET_CMD/__USER__/$USER}"
if bash -c "$CMD" >/dev/null 2>&1; then
  STATUS="OK"
  echo "[$(ts)] OK reset-usage ${USER}" | tee -a "$LOG"
else
  STATUS="FAIL"
  echo "[$(ts)] FAIL reset-usage ${USER}" | tee -a "$LOG"
fi

SUMMARY=$(cat <<EOF
Mode  : ${MODE}
Time  : $(ts)
User  : ${USER}
State : ${STATUS}
Log   : ${LOG}
EOF
)
echo "$SUMMARY" | tee -a "$LOG"
send_tg "NoobzVPN - Reset Usage (${MODE})" "$SUMMARY"
BASH
sudo chmod +x /usr/local/sbin/noobz_reset_usage_run

sudo tee /etc/systemd/system/noobz-reset@.service >/dev/null <<'UNIT'
[Unit]
Description=NoobzVPN reset usage for %I
After=network-online.target
Wants=network-online.target

[Service]
Type=oneshot
# Instance name kita pakai pola USER__MODE (double underscore)
ExecStart=/usr/local/sbin/noobz_reset_usage_run %I %I
User=root
Group=root
UNIT

sudo tee /usr/local/sbin/nvpn_schedule_reset >/dev/null <<'BASH'
#!/usr/bin/env bash
# Manage per-user systemd timer berdasarkan issued time user.
# Usage:
#   nvpn_schedule_reset add <user> <daily|monthly|yearly>
#   nvpn_schedule_reset remove <user> <daily|monthly|yearly>
#   nvpn_schedule_reset run <user> <daily|monthly|yearly>
#   nvpn_schedule_reset remove-user <user>
#   nvpn_schedule_reset purge
#   nvpn_schedule_reset list

set -euo pipefail

need(){ command -v "$1" >/dev/null 2>&1 || { echo "Need: $1"; exit 1; }; }
need noobzvpns
need systemctl

UNIT_DIR="/etc/systemd/system"

issued_of(){
  local u="$1"
  noobzvpns print "$u" 2>/dev/null \
    | sed -nE 's/^[[:space:]]*-issued[[:space:]]*:[[:space:]]*(.*)$/\1/p' \
    | head -n1
}

oncalendar_from(){
  local issued="$1" mode="$2"
  local part_date="${issued%% *}" part_time="${issued##* }"
  local y m d H M S
  IFS='-' read -r y m d <<<"$part_date"
  IFS=':' read -r H M S <<<"$part_time"

  case "$mode" in
    daily)   echo "*-*-* ${H}:${M}:${S}" ;;
    monthly) echo "*-*-${d} ${H}:${M}:${S}" ;;
    yearly)  echo "*-${m}-${d} ${H}:${M}:${S}" ;;
    *) echo ""; return 1 ;;
  esac
}

inst_name(){
  local u="$1" mode="$2"
  local safe="${u//[^a-zA-Z0-9_.-]/_}"
  echo "${safe}__${mode}"
}

make_timer(){
  local instance="$1" oncal="$2"
  local timer="${UNIT_DIR}/noobz-reset@${instance}.timer"
  cat > "$timer" <<TIMER
[Unit]
Description=Timer for noobz-reset@${instance}

[Timer]
OnCalendar=${oncal}
Persistent=true
RandomizedDelaySec=0

[Install]
WantedBy=timers.target
TIMER
  echo "$timer"
}

remove_instance(){
  local inst="$1"
  local unit_timer="noobz-reset@${inst}.timer"
  systemctl disable --now "$unit_timer" >/dev/null 2>&1 || true
  rm -f "${UNIT_DIR}/${unit_timer}" 2>/dev/null || true
}

cmd="${1:-}"
case "$cmd" in
  add)
    [[ $# -eq 3 ]] || { echo "Usage: $0 add <user> <daily|monthly|yearly>"; exit 1; }
    user="$2"; mode="$3"
    iss="$(issued_of "$user")" || true
    [[ -n "$iss" ]] || { echo "Issued time for user '$user' not found."; exit 1; }
    oncal="$(oncalendar_from "$iss" "$mode")"
    [[ -n "$oncal" ]] || { echo "Invalid mode."; exit 1; }
    inst="$(inst_name "$user" "$mode")"
    timer_file="$(make_timer "$inst" "$oncal")"
    systemctl daemon-reload
    systemctl enable --now "noobz-reset@${inst}.timer"
    echo "Added & started: noobz-reset@${inst}.timer (OnCalendar=${oncal})"
    ;;

  remove)
    [[ $# -eq 3 ]] || { echo "Usage: $0 remove <user> <daily|monthly|yearly>"; exit 1; }
    user="$2"; mode="$3"
    inst="$(inst_name "$user" "$mode")"
    remove_instance "$inst"
    systemctl daemon-reload
    echo "Removed: noobz-reset@${inst}.timer"
    ;;

  run)
    [[ $# -eq 3 ]] || { echo "Usage: $0 run <user> <daily|monthly|yearly>"; exit 1; }
    user="$2"; mode="$3"
    inst="$(inst_name "$user" "$mode")"
    systemctl start "noobz-reset@${inst}.service" || {
      echo "Direct run fallback..."
      /usr/local/sbin/noobz_reset_usage_run "$user" "$mode"
    }
    ;;

  remove-user)
    [[ $# -eq 2 ]] || { echo "Usage: $0 remove-user <user>"; exit 1; }
    user="$2"
    removed=0
    for mode in daily monthly yearly; do
      inst="$(inst_name "$user" "$mode")"
      if [[ -f "${UNIT_DIR}/noobz-reset@${inst}.timer" ]]; then
        remove_instance "$inst"
        echo "Removed: noobz-reset@${inst}.timer"
        removed=$((removed+1))
      fi
    done
    systemctl daemon-reload
    echo "Done. Removed ${removed} timer(s) for user '${user}'."
    ;;

  purge)
    # Hapus SEMUA timer noobz-reset@*.timer
    found=0
    shopt -s nullglob
    for f in "${UNIT_DIR}"/noobz-reset@*.timer; do
      base="$(basename "$f")"              # noobz-reset@<inst>.timer
      inst="${base#noobz-reset@}"
      inst="${inst%.timer}"
      remove_instance "$inst"
      echo "Removed: ${base}"
      found=$((found+1))
    done
    shopt -u nullglob
    systemctl daemon-reload
    echo "Purge complete. Removed ${found} timer(s)."
    ;-

  list)
    systemctl list-timers --all | grep -E 'noobz-reset@.*\.timer' || true
    ;;

  *)
    echo "Usage:"
    echo "  $0 add <user> <daily|monthly|yearly>"
    echo "  $0 remove <user> <daily|monthly|yearly>"
    echo "  $0 run <user> <daily|monthly|yearly>"
    echo "  $0 remove-user <user>"
    echo "  $0 purge"
    echo "  $0 list"
    exit 1
    ;;
esac
BASH
chmod +x /usr/local/sbin/nvpn_schedule_reset
systemctl daemon-reload

# Restart Marzban agar Xray reload xray_config.json
if docker ps --format '{{.Names}}' | grep -qi marzban; then
  echo "[marzban] restart container…"
cd /opt/marzban
docker compose down && docker compose up -d
cd
else
  echo "[marzban] (lewati restart docker: container bernama 'marzban' tidak terdeteksi)"
fi

echo
echo "=== DONE ==="
echo "- Xray extra config: ${XRAY_EXTRA}"
echo "  -> inbound socks-in @ ${XRAY_SOCKS_IP}:${XRAY_SOCKS_PORT}"
echo
echo "Tes cepat:"
echo "  ss -lnpt 'sport = :${XRAY_SOCKS_PORT}'          # host, pastikan Xray listen"
echo "  ip netns exec ${NS} bash -lc 'exec 3<>/dev/tcp/${XRAY_SOCKS_IP}/${XRAY_SOCKS_PORT} && echo OK || echo FAIL'"
echo "  tail -f /var/lib/marzban/assets/access.log | grep socks-in"
