#!/usr/bin/env bash
set -euo pipefail

SECRET_FILE="/opt/nvpn-sub/secret"
DOMAIN_FILE="/root/domain"

# Buat secret default kalau belum ada
if [[ ! -s "$SECRET_FILE" ]]; then
  echo "ganti-rahasia-yg-susah" > "$SECRET_FILE"
fi

if [[ $# -lt 1 ]]; then
  echo "Usage: $(basename "$0") USERNAME [days_valid|0] [withpass(0|1)]" >&2
  exit 1
fi

USER="$1"
DAYS="${2:-30}"
WITHPASS="${3:-0}"   # 1 = reveal password

# Baca SECRET & DOMAIN, buang newline agar bersih
SECRET="$(tr -d '\r\n' < "$SECRET_FILE")"
if [[ -s "$DOMAIN_FILE" ]]; then
  DOMAIN="$(tr -d ' \t\r\n' < "$DOMAIN_FILE")"
else
  DOMAIN="$(hostname -f 2>/dev/null || hostname)"
fi

# Hitung exp (0 = unlimited)
if [[ "$DAYS" == "0" ]]; then
  EXP=0
else
  EXP=$(( $(date +%s) + DAYS*24*3600 ))
fi

# Scope (pw = kirim password)
SCOPE=""
if [[ "$WITHPASS" == "1" ]]; then
  SCOPE="pw"
fi

# ======== FIX PENTING: pakai KOMA untuk string yang di-HMAC =========
# API memverifikasi HMAC atas: "username,exp,scope"
MSG="${USER},${EXP},${SCOPE}"

# HMAC-SHA256 (hexdigest) dengan secret apa adanya (tanpa base64-decode)
MAC="$(printf '%s' "$MSG" | openssl dgst -sha256 -hmac "$SECRET" -r | awk '{print $1}')"

# Plain token string pakai KOMA juga
PLAIN="${USER},${EXP},${SCOPE},${MAC}"

# Base64 URL-safe, tanpa newline & tanpa padding
if base64 --help 2>&1 | grep -q -- '-w'; then
  B64="$(printf '%s' "$PLAIN" | base64 -w0)"
else
  # fallback jika base64 tidak support -w0
  B64="$(printf '%s' "$PLAIN" | base64 | tr -d '\n')"
fi
TOKEN="$(printf '%s' "$B64" | tr '+/' '-_' | tr -d '=')"

URL="https://${DOMAIN}/nvpn/${TOKEN}"
echo "$URL"
