#!/usr/bin/env python3
import os, json, base64, hmac, hashlib, time
from datetime import datetime
from flask import Flask, jsonify, abort

app = Flask(__name__)

# --- Secret source: file first, then ENV ---
SECRET_FILE = os.environ.get("NVPN_SECRET_FILE", "/opt/nvpn-sub/secret")
_secret_txt = ""
if os.path.exists(SECRET_FILE):
    with open(SECRET_FILE, "r", encoding="utf-8") as f:
        _secret_txt = f.read().strip()          # strip newline/space
if not _secret_txt:
    _secret_txt = os.environ.get("NVPN_SECRET", "").strip()
if not _secret_txt:
    raise RuntimeError("NVPN secret not found (set NVPN_SECRET_FILE or NVPN_SECRET)")
NVPN_SECRET = _secret_txt.encode("utf-8")

DB_PATH       = os.environ.get("NVPN_DB", "/etc/noobzvpns/db_user.json")
SCHED_DIR     = os.environ.get("NVPN_SCHED_DIR", "/etc/noobzvpns/schedules")  # <— PATCH

def b64decode_urlsafe(b: str) -> bytes:
    return base64.urlsafe_b64decode(b + "=" * (-len(b) % 4))

def _hmac_hex(msg: str) -> str:
    return hmac.new(NVPN_SECRET, msg.encode("utf-8"), hashlib.sha256).hexdigest()

def decode_token(token: str):
    """
    Token = base64url("username,exp,scope,hexsig")
    Back-compat: terima juga varian lama yang menghitung HMAC pakai '.' sebagai separator.
    """
    try:
        raw = b64decode_urlsafe(token).decode("utf-8")
        parts = raw.split(",")
        if len(parts) != 4:
            return None, "malformed token"
        username, exp, scope, hexsig = parts

        # expiry check
        if exp and exp.isdigit() and int(exp) < int(time.time()):
            return None, "token expired"

        # cek HMAC: versi koma dulu
        if hmac.compare_digest(_hmac_hex(f"{username},{exp},{scope}"), hexsig):
            return {"username": username, "exp": int(exp) if exp.isdigit() else 0, "scope": scope}, None
        # fallback: versi titik
        if hmac.compare_digest(_hmac_hex(f"{username}.{exp}.{scope}"), hexsig):
            return {"username": username, "exp": int(exp) if exp.isdigit() else 0, "scope": scope}, None

        return None, "bad signature"
    except Exception as e:
        return None, f"decode error: {e}"

def load_db():
    with open(DB_PATH, "r", encoding="utf-8") as f:
        return json.load(f)

# --------- PATCH: baca jadwal dari file per-user ----------
def load_schedule(username: str) -> dict:
    """
    Baca /etc/noobzvpns/schedules/<user>.json (atau NVPN_SCHED_DIR).
    Kembalikan isi file apa adanya, dengan normalisasi strategy -> lowercase.
    Contoh file:
    {
      "username": "linggar1",
      "strategy": "daily",
      "issued": "2025-09-27 12:06:08",
      "issued_ts": 1758949568,
      "next_run": "2025-09-28 18:06:08 WIB",
      "next_run_ts": 1759057568
    }
    """
    try:
        p = os.path.join(SCHED_DIR, f"{username}.json")
        if not os.path.exists(p):
            return {}
        with open(p, "r", encoding="utf-8") as f:
            obj = json.load(f)
        if isinstance(obj.get("strategy"), str):
            obj["strategy"] = obj["strategy"].lower()
        return obj
    except Exception as e:
        app.logger.warning(f"schedule load error for {username}: {e}")
        return {}
# ----------------------------------------------------------

def normalize_user(username: str, node: dict, scope: str) -> dict:
    u = {
        "username": username,
        "blocked": bool(node.get("blocked", False)),
        "expired": node.get("expired", 0),
        "bandwidth": node.get("bandwidth", 0),  # GB
        "devices": node.get("devices", 0),
        "issued": node.get("issued", "-"),
        "statistic": {
            "bytes_usage": {
                "up":   int(node.get("statistic", {}).get("bytes_usage", {}).get("up", 0)),
                "down": int(node.get("statistic", {}).get("bytes_usage", {}).get("down", 0))
            },
            "active_devices": node.get("statistic", {}).get("active_devices", []) or []
        }
    }
    u["password"] = node.get("password", "") if ("pw" in (scope or "")) else ""
    return u

@app.route("/api/ping")
def ping():
    return jsonify({"ok": True, "ts": int(time.time())})

@app.route("/api/user/<token>.json")
def user_info(token):
    parsed, err = decode_token(token)
    if err:
        abort(401, description=err)
    username = parsed["username"]
    scope    = parsed.get("scope","")

    try:
        db = load_db()
        node = db.get("users", {}).get(username)
        if not node:
            abort(404, description="user not found")
    except Exception as e:
        abort(500, description=f"db error: {e}")

    # PATCH: sertakan schedule
    schedule = load_schedule(username)

    return jsonify({
        "user": normalize_user(username, node, scope),
        "schedule": schedule,  # <— PATCH
        "server_time": datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    })

if __name__ == "__main__":
    app.run(host="127.0.0.1", port=9001)
