#!/usr/bin/env python3
# subsrv.py — Subscription mini server (HTML + JSON) untuk cek quota, expired, & IP aktif dari Xray
# - Baca /usr/local/etc/xray/database.json (persist data user)
# - Baca /var/lib/marzban/assets/access.log (deteksi IP aktif dalam window waktu)
# - Proteksi dengan token HMAC berbasis SUB_SECRET (query ?token=...)
# - Endpoint dasar bisa diubah: SUB_BASE_HTML (default "s22"), SUB_BASE_JSON (default "s22j")
# - Zona waktu tampilan: TZ_OFFSET_HOURS (default 7 = WIB)
# - Window IP aktif: SUB_WINDOW_SEC (default 600 detik)

import os
import hmac
import hashlib
import json
import time
import re
from datetime import datetime, timezone, timedelta
from flask import Flask, request, jsonify, abort, render_template_string

# ========= KONFIGURASI via ENV =========
DB_PATH = os.environ.get("XRAY_DB", "/data/database.json")
ACCESS_LOG = os.environ.get("XRAY_ACCESS_LOG", "/logs/access.log")
WINDOW_SEC = int(os.environ.get("SUB_WINDOW_SEC", "600"))        # 10 menit
SECRET_KEY = os.environ.get("SUB_SECRET", "GANTI_INI_PANJANG_BANGET_32-64CHAR")   # GANTI di production
TZ = timezone(timedelta(hours=int(os.environ.get("TZ_OFFSET_HOURS", "7"))))  # WIB default
BASE_HTML = os.environ.get("SUB_BASE_HTML", "s22")   # path HTML → /s22/<email>
BASE_JSON = os.environ.get("SUB_BASE_JSON", "s22j")  # path JSON → /s22j/<email>

# ========= APP =========
app = Flask(__name__)

def hmac_token(email: str) -> str:
    return hmac.new(SECRET_KEY.encode(), email.encode(), hashlib.sha256).hexdigest()

def verify(email: str, token: str) -> bool:
    return hmac.compare_digest(hmac_token(email), token or "")

def load_db():
    with open(DB_PATH, "r") as f:
        return json.load(f)

def fmt_bytes(n: int) -> str:
    units = ["B","KB","MB","GB","TB","PB"]
    x = float(n); i = 0
    while x >= 1024 and i < len(units)-1:
        x /= 1024; i += 1
    return f"{x:.2f} {units[i]}"

# IP privat/lokal
_ipv4_priv = re.compile(r"^(127\.|10\.|192\.168\.|172\.(1[6-9]|2[0-9]|3[0-1])\.)")
def is_private_ip(ip: str) -> bool:
    ip_l = ip.lower()
    if ip in ("0.0.0.0", "localhost", "::1"):
        return True
    if _ipv4_priv.match(ip):
        return True
    # IPv6 heuristik
    if ip_l.startswith("fc") or ip_l.startswith("fd"):  # fc00::/7
        return True
    if ip_l.startswith("fe8"):                          # fe80::/10
        return True
    return False

# Parsing access.log Xray:
# Contoh baris:
# 2025/09/29 12:48:16.850496 from 36.81.183.133:0 accepted 1.1.1.1:853 [SSWS >> direct] email: linggar
# 2025/09/29 06:00:03.367871 from 112.215.220.20:0 accepted tcp:bing.com:80 [VLESS_WS >> direct] email: 276.Erlansaja
_ts_re = re.compile(r"^(\d{4}/\d{2}/\d{2})\s+(\d{2}):(\d{2}):(\d{2})")
_from_re = re.compile(r"\sfrom\s([^\s]+)")
def extract_ips_from_log(email: str, window_sec: int):
    out = {}
    if not os.path.isfile(ACCESS_LOG):
        return out

    # Asumsi waktu di log = waktu lokal server dengan offset TZ_OFFSET_HOURS
    # → Kita ubah ke UTC agar bisa dibandingkan dengan "now" UTC.
    cutoff_utc = datetime.now(timezone.utc) - timedelta(seconds=window_sec)
    tag = f"email: {email}"

    with open(ACCESS_LOG, "r", errors="ignore") as f:
        for line in f:
            if tag not in line:
                continue

            m = _ts_re.match(line)
            if not m:
                continue
            try:
                # Timestamp tanpa mikrodetik → "%Y/%m/%d %H:%M:%S"
                ts_local = datetime.strptime(
                    f"{m.group(1)} {m.group(2)}:{m.group(3)}:{m.group(4)}",
                    "%Y/%m/%d %H:%M:%S"
                )
            except Exception:
                continue
            # log diasumsikan dalam TZ lokal (TZ env). Convert ke UTC
            ts_utc = ts_local.replace(tzinfo=TZ).astimezone(timezone.utc)
            if ts_utc < cutoff_utc:
                continue

            mf = _from_re.search(line)
            if not mf:
                continue
            host = mf.group(1)  # contoh "36.81.183.133:0" atau "::1:12345"
            # buang port: ambil sebelum ":" terakhir (untuk IPv6 tanpa bracket ini heuristik paling aman)
            if ":" in host:
                ip = host.rsplit(":", 1)[0]
            else:
                ip = host
            ip = ip.rstrip(",;")

            if is_private_ip(ip):
                continue

            out[ip] = max(out.get(ip, 0), int(ts_utc.timestamp()))
    return out

def user_view(email: str):
    db = load_db()
    user = db.get("users", {}).get(email)
    if not user:
        return None
    quota = int(user.get("quota_bytes", 0))
    used = int(user.get("used_bytes", 0))
    remaining = max(quota - used, 0)
    expire_at = int(user.get("expire_at", 0))
    now = int(time.time())
    expire_in_s = max(expire_at - now, 0) if expire_at > 0 else 0
    expire_in_days = (expire_in_s + 86399) // 86400 if expire_in_s > 0 else 0

    created_at = user.get("created_at", "")
    # Format WIB
    expire_str = datetime.fromtimestamp(expire_at, TZ).strftime("%Y-%m-%d %H:%M:%S WIB") if expire_at > 0 else "-"
    created_str = created_at
    try:
        created_dt = datetime.strptime(created_at, "%Y-%m-%dT%H:%M:%SZ").replace(tzinfo=timezone.utc).astimezone(TZ)
        created_str = created_dt.strftime("%Y-%m-%d %H:%M:%S WIB")
    except Exception:
        pass

    return {
        "email": email,
        "enabled": bool(user.get("enabled", False)),
        "quota_bytes": quota,
        "used_bytes": used,
        "remaining_bytes": remaining,
        "quota_h": fmt_bytes(quota),
        "used_h": fmt_bytes(used),
        "remaining_h": fmt_bytes(remaining),
        "expire_at": expire_at,
        "expire_in_days": expire_in_days,
        "expire_wib": expire_str,
        "created_wib": created_str,
    }

HTML_TPL = """
<!doctype html>
<html>
<head>
  <meta charset="utf-8"/><meta name="viewport" content="width=device-width, initial-scale=1"/>
  <title>Subscription · {{email}}</title>
  <style>
    body{font-family:ui-sans-serif,-apple-system,Inter,Segoe UI,Roboto,Arial;max-width:680px;margin:40px auto;padding:0 16px;color:#e5e7eb;background:#0b1220}
    .card{background:#0f172a;border:1px solid #1f2937;border-radius:14px;padding:16px 18px;box-shadow:0 4px 20px rgba(0,0,0,.25)}
    h1{font-size:20px;margin:0 0 12px;color:#e2e8f0}
    .row{display:flex;justify-content:space-between;padding:10px 0;border-bottom:1px dashed #334155}
    .row:last-child{border-bottom:none}
    .k{color:#93c5fd}
    .v{color:#e5e7eb}
    .ip{font-family:ui-monospace,Menlo,Consolas,monospace;color:#d1fae5}
    .muted{color:#94a3b8;font-size:12px}
  </style>
</head>
<body>
  <div class="card">
    <h1>ShadowSocks · {{email}}</h1>
    <div class="row"><div class="k">Status</div><div class="v">{{ 'Aktif' if enabled else 'Nonaktif' }}</div></div>
    <div class="row"><div class="k">Kuota</div><div class="v">{{ used_h }} / {{ quota_h }} <span class="muted">(Sisa {{ remaining_h }})</span></div></div>
    <div class="row"><div class="k">Expired</div><div class="v">{{ expire_wib }} <span class="muted">(~{{ expire_in_days }} hari)</span></div></div>
    <div class="row"><div class="k">Dibuat</div><div class="v">{{ created_wib }}</div></div>
    <div class="row"><div class="k">IP aktif ({{ ips|length }})</div>
      <div class="v">
        {% if ips %}{% for ip, ts in ips %}<div class="ip">{{ip}}</div>{% endfor %}{% else %}<span class="muted">Tidak ada koneksi aktif ({{window}}s)</span>{% endif %}
      </div>
    </div>
  </div>
  <p class="muted" style="text-align:center;margin-top:12px">Window IP: {{window}} detik · Zona waktu: WIB</p>
</body>
</html>
"""

@app.get(f"/{BASE_JSON}/<email>")
def api(email):
    token = request.args.get("token", "")
    if not verify(email, token):
        abort(403)
    info = user_view(email)
    if not info:
        abort(404)
    ips = extract_ips_from_log(email, WINDOW_SEC)
    info["online_ips"] = sorted(ips.keys())
    info["online_ip_count"] = len(info["online_ips"])
    info["window_sec"] = WINDOW_SEC
    return jsonify(info)

@app.get(f"/{BASE_HTML}/<email>")
def page(email):
    token = request.args.get("token", "")
    if not verify(email, token):
        abort(403)
    info = user_view(email)
    if not info:
        abort(404)
    ips = extract_ips_from_log(email, WINDOW_SEC)
    ips_sorted = sorted(ips.items(), key=lambda kv: kv[1], reverse=True)
    return render_template_string(
        HTML_TPL,
        email=email,
        enabled=info["enabled"],
        used_h=info["used_h"],
        quota_h=info["quota_h"],
        remaining_h=info["remaining_h"],
        expire_wib=info["expire_wib"],
        expire_in_days=info["expire_in_days"],
        created_wib=info["created_wib"],
        ips=ips_sorted,
        window=WINDOW_SEC,
    )

@app.get("/")
def root():
    return jsonify({
        "ok": True,
        "usage": f"/{BASE_HTML}/<email>?token=...  or  /{BASE_JSON}/<email>?token=...",
        "window_sec": WINDOW_SEC,
        "tz_offset_hours": int(TZ.utcoffset(datetime.now()).total_seconds()/3600),
        "base_html": BASE_HTML,
        "base_json": BASE_JSON,
    })

if __name__ == "__main__":
    # Dengarkan di semua interface (compose kita host-mode, atau publish port)
    app.run(host="127.0.0.1", port=int(os.environ.get("PORT","8989")))
