#!/usr/bin/env python3
# cekservice.py - Ringkasan layanan LingVPN x Marzban (Python)
# deps: apt install -y python3-requests
import os, re, json, subprocess, sys
from pathlib import Path

try:
    import requests
except Exception:
    print("Module 'requests' belum terpasang. Jalankan: apt install -y python3-requests", file=sys.stderr)
    sys.exit(1)

# ====== Konfigurasi dasar ======
DOMAIN_FILE = "/root/domain"
TOKEN_FILE  = "/root/token.json"
ENV_FILE    = "/opt/marzban/.env"
DB_SSH_PATH = "/etc/lingvpn/db.json"     # database user SSH-WS
NGINX_PORT  = 8081
XRAY_TLS    = 443

def read_domain():
    try:
        return Path(DOMAIN_FILE).read_text().strip()
    except Exception:
        return "example.id"

def read_token():
    try:
        data = json.loads(Path(TOKEN_FILE).read_text())
        return data.get("access_token") or data.get("token") or ""
    except Exception:
        return ""

def get_marzban_api_port():
    port = "7879"
    try:
        for line in Path(ENV_FILE).read_text().splitlines():
            if line.strip().startswith("UVICORN_PORT"):
                m = re.search(r"=\s*['\"]?(\d+)['\"]?", line)
                if m:
                    port = m.group(1); break
    except Exception:
        pass
    return port

API_HOST = os.environ.get("API_HOST", "127.0.0.1")
API_PORT = os.environ.get("API_PORT", get_marzban_api_port())
API_BASE = f"http://{API_HOST}:{API_PORT}/api"
TOKEN    = read_token()

# ====== Warna & ASCII ======
def colorize(s, c):
    if not sys.stdout.isatty():
        return s
    C = {
        "red":"\033[0;31m","green":"\033[0;32m","yellow":"\033[0;33m",
        "blue":"\033[0;34m","magenta":"\033[0;35m","cyan":"\033[0;36m",
        "white":"\033[0;37m","reset":"\033[0m"
    }
    return f"{C.get(c,'')}{s}{C['reset']}"

def hr():
    return colorize("─"*48, "cyan")

def panel(title):
    print(colorize("━"*48, "cyan"))
    print(colorize(f"[ {title} ]".center(48), "blue"))
    print(colorize("━"*48, "cyan"))

def okfail(ok):
    return colorize("Okay", "green") if ok else colorize("Not Okay","red")

# key/value printer dengan label rata kiri & titik dua sejajar
_ansi_re = re.compile(r"\x1b\[[0-9;]*m")
def kv(label, value, lw=24):
    vis = len(_ansi_re.sub("", label))
    pad = max(0, lw - vis)
    print(f"{label}{' ' * pad} : {value}")

# ====== Shell helpers ======
def run(cmd, shell=False):
    try:
        if shell:
            return subprocess.run(cmd, shell=True, text=True, capture_output=True)
        else:
            return subprocess.run(cmd, text=True, capture_output=True)
    except Exception as e:
        return subprocess.CompletedProcess(cmd, 1, "", str(e))

def systemd_active(unit):
    return subprocess.call(["systemctl","is-active","--quiet",unit]) == 0

def listener_present(port):
    r = run("ss -ltn 2>/dev/null | awk '{print $4}'", shell=True)
    if r.returncode == 0 and r.stdout:
        for ln in r.stdout.splitlines():
            if re.search(rf'[:\.]{port}$', ln.strip()):
                return True
    r = run("netstat -ltn 2>/dev/null | awk '{print $4}'", shell=True)
    if r.returncode == 0 and r.stdout:
        for ln in r.stdout.splitlines():
            if re.search(rf'[:\.]{port}$', ln.strip()):
                return True
    return False

# ====== Docker compose tag (gozargah/marzban) ======
def marzban_image_tag():
    try:
        yml = Path("/opt/marzban/docker-compose.yml").read_text()
        m = re.search(r'image:\s*gozargah/marzban:([^\s]+)', yml)
        if m: return m.group(1).strip()
    except Exception:
        pass
    return "-"

# ====== Marzban API ======
def bearer():
    return {"Authorization": f"Bearer {TOKEN}", "accept":"application/json"} if TOKEN else {"accept":"application/json"}

def get_system_info():
    try:
        r = requests.get(f"{API_BASE}/system", headers=bearer(), timeout=10)
        r.raise_for_status()
        return r.json()
    except Exception:
        return {}

def get_core_version():
    try:
        r = requests.get(f"{API_BASE}/core", headers=bearer(), timeout=10)
        if r.ok:
            return (r.json() or {}).get("version") or "Unknown"
    except Exception:
        pass
    return "Unknown"

def count_users():
    limit = 200
    offset = 0
    agg = {"total":0, "active":0, "disabled":0, "expired":0, "limited":0,
           "vmess":0, "vless":0, "trojan":0, "shadowsocks":0}
    while True:
        try:
            r = requests.get(f"{API_BASE}/users", headers=bearer(),
                             params={"limit":limit,"offset":offset}, timeout=15)
            r.raise_for_status()
            data = r.json()
        except Exception:
            break
        users = data.get("users") or []
        if not users: break
        for u in users:
            agg["total"] += 1
            status = (u.get("status") or "").lower()
            if status in agg:
                agg[status] += 1
            proxies = u.get("proxies") or {}
            for p in ("vmess","vless","trojan","shadowsocks"):
                if p in proxies: agg[p] += 1
        if len(users) < limit: break
        offset += limit
    return agg

# ====== SSH-WS user count ======
def ssh_user_count():
    try:
        if not Path(DB_SSH_PATH).exists(): return 0
        db = json.loads(Path(DB_SSH_PATH).read_text() or "{}")
        users = db.get("users") or {}
        return len(users.keys())
    except Exception:
        return 0

# ====== Human bytes ======
def human_bytes(n):
    try:
        n = float(n or 0)
    except Exception:
        n = 0.0
    units = ["B","KB","MB","GB","TB","PB"]
    i = 0
    while n >= 1024.0 and i < len(units)-1:
        n /= 1024.0; i += 1
    return f"{int(n)} {units[i]}" if i==0 else f"{n:.2f} {units[i]}"

# ====== MAIN ======
def main():
    domain = read_domain()
    img_tag = marzban_image_tag()

    # status services
    docker_ok = systemd_active("docker")
    xray_ok   = listener_present(XRAY_TLS)
    nginx_ok  = listener_present(NGINX_PORT)
    ufw_ok    = systemd_active("ufw")
    marz_ok   = listener_present(API_PORT)

    # API pulls
    sysinfo   = get_system_info()
    xray_ver  = get_core_version()
    mz_ver    = sysinfo.get("version","-")

    mem_used  = human_bytes(sysinfo.get("mem_used",0))
    mem_total = human_bytes(sysinfo.get("mem_total",0))
    cpu_cores = sysinfo.get("cpu_cores","-")
    cpu_usage = sysinfo.get("cpu_usage","-")
    in_bw     = human_bytes(sysinfo.get("incoming_bandwidth",0))
    out_bw    = human_bytes(sysinfo.get("outgoing_bandwidth",0))
    in_spd    = human_bytes(sysinfo.get("incoming_bandwidth_speed",0)) + "/s"
    out_spd   = human_bytes(sysinfo.get("outgoing_bandwidth_speed",0)) + "/s"
    online    = sysinfo.get("online_users","-")

    # users
    agg = count_users()
    ssh_cnt = ssh_user_count()

    # ===== OUTPUT =====
    panel("Service Information")
    kv("Docker Container",          okfail(docker_ok))
    kv(f"Xray Core (:{XRAY_TLS})",  okfail(xray_ok))
    kv(f"Nginx (:{NGINX_PORT})",    okfail(nginx_ok))
    kv("Firewall (ufw)",            okfail(ufw_ok))
    kv(f"Marzban API (:{API_PORT})",okfail(marz_ok))
    print(hr())
    print(colorize(
        "SHARING PORT 443 MARZBAN VERSION AMAN SEMUA BOSSKUH" if xray_ok else "ADA ERROR BOSSKUH",
        "green" if xray_ok else "red"
    ))
    print(hr())

    panel("Marzban System Information")
    kv("Marzban Version",           f"{mz_ver}  ({img_tag})")
    kv("XrayCore Version",          xray_ver)
    kv("Memory Usage",              f"{mem_used}/{mem_total}")
    kv("CPU Cores",                 f"{cpu_cores} cores")
    kv("CPU Usage",                 f"{cpu_usage}%")
    kv("Inbound Bandwidth",         in_bw)
    kv("Outbound Bandwidth",        out_bw)
    kv("Inbound Speed",             in_spd)
    kv("Outbound Speed",            out_spd)
    print(hr())

    panel("Users per Protocol & Status")
    kv("Total Users (Xray)",        str(agg.get("total",0)))
    kv("  • VMess",                 str(agg.get('vmess',0)))
    kv("  • VLess",                 str(agg.get('vless',0)))
    kv("  • Trojan",                str(agg.get('trojan',0)))
    kv("  • Shadowsocks",           str(agg.get('shadowsocks',0)))
    print("")
    print("Status breakdown:")
    kv("  • Active",                colorize(str(agg.get('active',0)),'green'))
    kv("  • Disabled",              colorize(str(agg.get('disabled',0)),'yellow'))
    kv("  • Expired",               colorize(str(agg.get('expired',0)),'red'))
    kv("  • Limited",               colorize(str(agg.get('limited',0)),'magenta'))
    kv("  • Users Online",          colorize(str(online),'green'))
    print(hr())
    kv("Users SSH-WS (DB)",         str(ssh_cnt))
    print(hr())
    kv("Dashboard GUI",             f"https://{domain}/dashboard")

if __name__ == "__main__":
    if os.geteuid() != 0:
        print("Jalankan sebagai root.", file=sys.stderr); sys.exit(1)
    if not TOKEN:
        print("Warning: token kosong (/root/token.json). Buat token API untuk data lengkap.", file=sys.stderr)
    main()
