#!/usr/bin/env python3
# LingVPN Enforcer – clearlog → wait 5m → scan → enforce (max-dev & torrent)
# - Menonaktifkan user yang melebihi IP limit ATAU terdeteksi TORRENT di access.log
# - Lock durasi = /var/lib/marzban/penalty_time.conf (menit), default 60
# - Siklus aman: pertama jalan clear access.log → set pending 5m → exit
#   setelah 5m baru scan log “fresh”
# - Semua notifikasi Telegram diberi tag [domain]
# - OPTIONAL: export LINGVPN_CLEAR_SYSTEM_LOGS=1 untuk ikut bersihkan log sistem

import os, re, sys, time, json, pathlib, socket, glob
from datetime import datetime
import requests

# ---------- Paths & const ----------
DATA_DIR      = "/var/lib/marzban"
LOG_FILE      = f"{DATA_DIR}/assets/access.log"
META_DIR      = f"{DATA_DIR}/meta"
LOCK_DIR      = f"{DATA_DIR}/locks"
MAXDEV_PATH   = f"{META_DIR}/max_devices.json"
DEFAULT_MAX   = f"{DATA_DIR}/max_ips.conf"      # integer
PENALTY_PATH  = f"{DATA_DIR}/penalty_time.conf" # minutes
TG_CONF       = "/etc/autokill/telegram_config.conf"  # botToken=.., chatId=..
PENDING_GLOBAL= f"{DATA_DIR}/pending_global.json"     # { "until": ts }

API_HOST = os.environ.get("API_HOST", "127.0.0.1")
RECHECK_WAIT_SEC = 5 * 60  # 5 menit
CLEAR_SYSTEM_LOGS = os.environ.get("LINGVPN_CLEAR_SYSTEM_LOGS", "0") in ("1","true","yes")

# ---------- Helpers ----------
def is_private_ip(ip):
    """Check if IP is private/local."""
    try:
        parts = ip.split('.')
        if len(parts) != 4:
            return True
        a = int(parts[0])
        b = int(parts[1])
        # Private ranges: 10.x, 172.16-31.x, 192.168.x, 127.x, 169.254.x
        if a == 10 or a == 127:
            return True
        if a == 172 and 16 <= b <= 31:
            return True
        if a == 192 and b == 168:
            return True
        if a == 169 and b == 254:
            return True
        return False
    except Exception:
        return True

def ensure_dirs():
    for d in (META_DIR, LOCK_DIR, os.path.dirname(LOG_FILE)):
        pathlib.Path(d).mkdir(parents=True, exist_ok=True)

def get_domain():
    p = "/root/domain"
    try:
        if os.path.exists(p):
            d = open(p,"r",encoding="utf-8").read().strip()
            if d: return d
    except Exception: pass
    return socket.gethostname()

def get_api_port():
    port = os.environ.get("API_PORT")
    if port: return port
    try:
        for line in open("/opt/marzban/.env","r",encoding="utf-8"):
            if line.strip().startswith("UVICORN_PORT"):
                m = re.search(r"=\s*['\"]?(\d+)['\"]?", line)
                if m: return m.group(1)
    except Exception:
        pass
    return "7879"

API_PORT = get_api_port()
API_BASE = f"http://{API_HOST}:{API_PORT}/api"

def bearer():
    try:
        tok = json.load(open("/root/token.json","r"))
        t = tok.get("access_token") or tok.get("token") or ""
    except Exception:
        t = ""
    if not t:
        print("enforcer: token tidak ditemukan /root/token.json", file=sys.stderr)
        sys.exit(1)
    return {"Authorization": f"Bearer {t}",
            "accept": "application/json",
            "Content-Type": "application/json"}

def load_maxdev_map():
    try: return json.load(open(MAXDEV_PATH,"r",encoding="utf-8"))
    except Exception: return {}

def load_default_max():
    try: return int(open(DEFAULT_MAX,"r").read().strip())
    except Exception: return 3

def load_penalty_minutes():
    try: return int(open(PENALTY_PATH,"r").read().strip())
    except Exception: return 60

def load_tg():
    if not os.path.exists(TG_CONF): return None, None
    bot = chat = None
    for line in open(TG_CONF,"r",encoding="utf-8"):
        line=line.strip()
        if line.startswith("botToken="): bot=line.split("=",1)[1].strip()
        if line.startswith("chatId="):   chat=line.split("=",1)[1].strip()
    return bot, chat

def tg_send(text):
    bot, chat = load_tg()
    if not bot or not chat: return
    try:
        requests.get(f"https://api.telegram.org/bot{bot}/sendMessage",
                     params={"chat_id": chat, "text": text}, timeout=10)
    except Exception:
        pass

def disable_user(uname):
    r = requests.put(f"{API_BASE}/user/{uname}", headers=bearer(),
                     data=json.dumps({"status":"disabled"}), timeout=20)
    return r.ok

def enable_user(uname):
    r = requests.put(f"{API_BASE}/user/{uname}", headers=bearer(),
                     data=json.dumps({"status":"active"}), timeout=20)
    return r.ok

def record_lock(uname, seconds, ips, count, reason="maxdev"):
    until = int(time.time()) + int(seconds)
    pathlib.Path(LOCK_DIR).mkdir(parents=True, exist_ok=True)
    with open(f"{LOCK_DIR}/{uname}.json","w",encoding="utf-8") as f:
        json.dump({"locked_until": until, "ips": ips, "count": count, "reason": reason}, f)

def read_lock(uname):
    p = f"{LOCK_DIR}/{uname}.json"
    if not os.path.exists(p): return None
    try: return json.load(open(p,"r",encoding="utf-8"))
    except Exception: return None

def clear_lock(uname):
    p = f"{LOCK_DIR}/{uname}.json"
    try:
        if os.path.exists(p): os.unlink(p)
    except Exception: pass

# ---------- Pending global gate (ganti “sleep 5m”) ----------
def pending_start():
    until = int(time.time()) + RECHECK_WAIT_SEC
    with open(PENDING_GLOBAL,"w",encoding="utf-8") as f:
        json.dump({"until": until}, f)

def pending_get():
    if not os.path.exists(PENDING_GLOBAL): return None
    try:
        return json.load(open(PENDING_GLOBAL,"r",encoding="utf-8"))
    except Exception:
        return None

def pending_clear():
    try:
        if os.path.exists(PENDING_GLOBAL): os.unlink(PENDING_GLOBAL)
    except Exception:
        pass

# ---------- Clear logs ----------
def clear_access_log_only():
    try:
        open(LOG_FILE,"w").close()
        return True
    except Exception:
        return False

def clear_system_logs_wide():
    patterns = [
        "/var/log/**/*.log", "/var/log/**/*.err", "/var/log/mail.*",
        "/var/lib/**/*.log","/var/lib/**/*.err","/var/lib/mail.*",
        "/var/log/syslog","/var/log/btmp","/var/log/messages","/var/log/debug",
    ]
    for pat in patterns:
        for p in glob.glob(pat, recursive=True):
            try: open(p,"w").close()
            except Exception: pass

# ---------- Log parsing ----------
IP_RE   = re.compile(r'(?<!\d)(?:\d{1,3}\.){3}\d{1,3}(?!\d)')
USER_PATTERNS = [
    re.compile(r'email[=:]\s*(?P<u>[A-Za-z0-9_\.~-]{3,64})'),
    re.compile(r'user[=:]\s*(?P<u>[A-Za-z0-9_\.~-]{3,64})'),
]

def fetch_user_list():
    try:
        r = requests.get(f"{API_BASE}/users", headers=bearer(),
                         params={"limit":2000,"offset":0}, timeout=20)
        r.raise_for_status()
        data = r.json()
        return data.get("users") or []
    except Exception:
        return []

def build_username_regex(usernames):
    esc = [re.escape(u) for u in usernames if u]
    if not esc: return None
    return re.compile(r'(?<![A-Za-z0-9_])(?:' + "|".join(esc) + r')(?![A-Za-z0-9_])')

def scan_connections_accurate(usernames):
    out = {u: set() for u in usernames}
    if not os.path.exists(LOG_FILE):
        return out

    name_re = build_username_regex(usernames)

    # Temp storage: username -> {subnet_key: [list_of_ips]}
    user_subnet_ips = {u: {} for u in usernames}

    try:
        with open(LOG_FILE, "r", errors="ignore") as f:
            for line in f:
                ipm = IP_RE.search(line)
                if not ipm:
                    continue
                ip = ipm.group(0)

                # Skip private IPs
                if is_private_ip(ip):
                    continue

                # Detect username
                u = None
                for pat in USER_PATTERNS:
                    m = pat.search(line)
                    if m:
                        cand = m.group("u")
                        if cand in out:
                            u = cand
                            break

                if u is None and name_re:
                    m2 = name_re.search(line)
                    if m2:
                        u = m2.group(0)

                if u and u in out:
                    # Group by subnet /16 (2 oktet pertama)
                    parts = ip.split(".")
                    if len(parts) == 4:
                        subnet_key = f"{parts[0]}.{parts[1]}"
                        if subnet_key not in user_subnet_ips[u]:
                            user_subnet_ips[u][subnet_key] = []
                        user_subnet_ips[u][subnet_key].append(ip)
    except Exception:
        pass

    # Ambil 1 IP representatif per subnet untuk setiap user
    for u in usernames:
        if u in user_subnet_ips:
            for subnet_key, ips in user_subnet_ips[u].items():
                # Ambil IP pertama (alphabetically) sebagai representatif
                representative_ip = sorted(ips)[0]
                out[u].add(representative_ip)

    return out

# --- NEW: deteksi TORRENT berbasis tag di access.log ---
def detect_torrent_offenders(usernames):
    """
    Kembalikan dict: username -> set(IP yang terlibat)
    Mencari baris yang mengandung 'TORRENT' (case-insensitive).
    Username dideteksi pakai USER_PATTERNS lalu fallback boundary regex.
    """
    offenders = {}
    if not os.path.exists(LOG_FILE): return offenders
    name_re = build_username_regex(usernames)
    try:
        with open(LOG_FILE, "r", errors="ignore") as f:
            for line in f:
                if "TORRENT" not in line.upper():
                    continue
                ipm = IP_RE.search(line)
                ip = ipm.group(0) if ipm else None

                u = None
                for pat in USER_PATTERNS:
                    m = pat.search(line)
                    if m:
                        cand = m.group("u")
                        if cand in usernames:
                            u = cand
                            break
                if u is None and name_re:
                    m2 = name_re.search(line)
                    if m2:
                        u = m2.group(0)

                if u:
                    s = offenders.setdefault(u, set())
                    if ip:
                        s.add(ip)
    except Exception:
        pass
    return offenders

# ---------- Main ----------
def main():
    ensure_dirs()
    domain = get_domain()
    penalty_min = load_penalty_minutes()
    penalty_sec = penalty_min * 60
    maxdev_map = load_maxdev_map()
    default_max = load_default_max()

    # Gate: clear → wait 5m
    pend = pending_get()
    if pend is None:
        ok = clear_access_log_only()
        if CLEAR_SYSTEM_LOGS:
            clear_system_logs_wide()
        pending_start()
        print(f"[enforcer] clear access.log: {ok}; start 5m window")
        return
    else:
        now = int(time.time())
        until = int(pend.get("until", now))
        if now < until:
            print(f"[enforcer] waiting recheck window: {until-now}s left")
            return
        # lanjut scan; pending dibersihkan di akhir

    # Ambil user list
    all_users = fetch_user_list()
    usernames = [u.get("username") for u in all_users if u.get("username")]

    # Scan koneksi & torrent
    user_ips = scan_connections_accurate(usernames)
    torrent_users = detect_torrent_offenders(usernames)

    # Enforce: TORRENT (SELALU disable, walau tidak melebihi max-dev)
    for uname, ips in torrent_users.items():
        lk = read_lock(uname)
        now_ts = int(time.time())
        if lk and int(lk.get("locked_until",0)) > now_ts:
            continue
        if disable_user(uname):
            record_lock(uname, penalty_sec, sorted(list(ips)), 0, reason="torrent")
            ip_str = ", ".join(sorted(list(ips))) if ips else "-"
            tg_send(f"[{domain}] 🚫 TORRENT: {uname} terdeteksi BitTorrent. Akun di-disable {penalty_min} menit. IPs: {ip_str}")
            print(f"Disabled (TORRENT) {uname} ips={ip_str}")
        else:
            print(f"Gagal disable (TORRENT) {uname}", file=sys.stderr)

    # Enforce: Max devices
    for u in all_users:
        uname = u.get("username")
        if not uname: continue
        ips = user_ips.get(uname, set())
        cnt = len(ips)
        lim = maxdev_map.get(uname, default_max)
        if lim is None or lim <= 0:
            continue
        if cnt > lim:
            lk = read_lock(uname)
            now_ts = int(time.time())
            if lk and int(lk.get("locked_until",0)) > now_ts:
                continue
            if disable_user(uname):
                record_lock(uname, penalty_sec, sorted(list(ips)), cnt, reason="maxdev")
                tg_send(f"[{domain}] ⚠️ ALERT: {uname} melebihi batas IP ({cnt}>{lim}). Akun di-disable {penalty_min} menit.")
                print(f"Disabled {uname} ({cnt}>{lim})")
            else:
                print(f"Gagal disable {uname}", file=sys.stderr)

    # Re-enable yang lock selesai
    now_ts = int(time.time())
    for u in all_users:
        uname = u.get("username"); status = (u.get("status") or "").lower()
        if not uname: continue
        lk = read_lock(uname)
        if not lk: 
            continue
        if now_ts >= int(lk.get("locked_until",0)):
            if status == "disabled":
                if enable_user(uname):
                    clear_lock(uname)
                    reason = lk.get("reason","lock")
                    tg_send(f"[{domain}] ✅ INFO: {uname} lock ({reason}) selesai, akun diaktifkan kembali.")
                    print(f"Re-enabled {uname}")
                else:
                    print(f"Gagal enable {uname}", file=sys.stderr)
            else:
                # user sudah active tapi file lock masih ada
                clear_lock(uname)

    pending_clear()
    print("[enforcer] cycle done, pending cleared.")

if __name__ == "__main__":
    main()
