#!/usr/bin/env python3
# lingvpn_menu_mz.py - Interactive TUI ala SSH-WS untuk Marzban (frontend lingvpn_mz.py)
# - Semua aksi memanggil /usr/local/sbin/lingvpn_mz.py
# - VLESS: add → lalu modify set flow xtls-rprx-vision (tanpa --proto agar tidak mengubah inbounds)
# - Notifikasi Telegram: HANYA untuk add, delete, extend, reset-usage, modify

import os, sys, json, subprocess, shlex, shutil, time, re, textwrap, base64, socket
from datetime import datetime, timezone
from typing import Optional  # kompat Python 3.8/3.9
from pathlib import Path
try:
    import requests
except Exception:
    requests = None

LVMZ = os.environ.get("LVMZ", "/usr/local/sbin/lingvpn_mz.py")
TG_CONF = "/etc/gegevps/bin/telegram_config.conf"
DOMAIN_FILE = "/root/domain"

# ================== Telegram ==================
def _load_tg():
    if not os.path.exists(TG_CONF):
        return None, None
    bot = chat = None
    try:
        for line in open(TG_CONF, "r", encoding="utf-8"):
            line = line.strip()
            if not line or line.startswith("#"): continue
            if line.startswith("TELEGRAM_BOT_TOKEN="): bot = line.split("=",1)[1].strip()
            if line.startswith("TELEGRAM_CHAT_ID="):   chat = line.split("=",1)[1].strip()
    except Exception:
        return None, None
    return bot, chat

def _tg_escape(s: str) -> str:
    return (s.replace("&", "&amp;")
             .replace("<", "&lt;")
             .replace(">", "&gt;"))

def tg_format_summary_html(summary: str) -> str:
    # bungkus nilai setelah ":" di baris-baris tertentu agar monospace
    mono_prefixes = ("UUID:", "Password:", "URI [TLS]:", "URI [nTLS]:")
    out = []
    for ln in summary.splitlines():
        if ln.startswith(mono_prefixes):
            k, _, v = ln.partition(":")
            out.append(f"{_tg_escape(k+': ')}<code>{_tg_escape(v.strip())}</code>")
        else:
            out.append(_tg_escape(ln))
    return "\n".join(out)

def tg_send(text, html: bool=False):
    bot, chat = _load_tg()
    if not bot or not chat:
        return
    try:
        args = ["curl", "-s", f"https://api.telegram.org/bot{bot}/sendMessage",
                "--data-urlencode", f"chat_id={chat}"]
        if html:
            args += ["--data-urlencode", "parse_mode=HTML",
                     "--data-urlencode", f"text={text}"]
        else:
            args += ["--data-urlencode", f"text={text}"]
        subprocess.run(args, stdout=subprocess.DEVNULL,
                       stderr=subprocess.DEVNULL, check=False)
    except Exception:
        pass

# ================== Utils ==================
def req_root():
    if os.geteuid() != 0:
        print("Jalankan sebagai root."); sys.exit(1)

def run_lvmz(args, capture=False):
    cmd = [sys.executable, LVMZ] + args
    if capture:
        return subprocess.run(cmd, text=True, capture_output=True)
    return subprocess.call(cmd)

def safe_input(prompt):
    try:
        return input(prompt)
    except (KeyboardInterrupt, EOFError):
        print("\n(dibatalkan)")
        return None

def pause():
    _ = safe_input("\n[Enter] untuk kembali...")

def human_bytes(n):
    try: n=float(n or 0)
    except: n=0.0
    units=["B","KB","MB","GB","TB","PB"]; i=0
    while n>=1024 and i<len(units)-1: n/=1024.0; i+=1
    return f"{int(n)} {units[i]}" if i==0 else f"{n:.2f} {units[i]}"

def ts_to_utc(ts):
    try:
        ts=int(ts or 0)
        if ts<=0: return "AlwaysON"
        return datetime.fromtimestamp(ts, tz=timezone.utc).strftime("%Y-%m-%d %H:%M:%SZ")
    except Exception:
        return str(ts)

def read_domain():
    try:
        if os.path.exists(DOMAIN_FILE):
            return open(DOMAIN_FILE,"r",encoding="utf-8").read().strip()
    except Exception:
        pass
    return "example.com"

RESET_LABELS = {
    "no_reset": "Loss_doll",
    "day":      "Harian",
    "week":     "Mingguan",
    "month":    "Bulanan",
    "year":     "Tahunan",
}
def reset_label(v):
    if not v: return "-"
    return RESET_LABELS.get(v, v)

# ================== ANSI panel ==================
_ANSI_RE = re.compile(r"\x1b\[[0-9;]*m")
def _ansi(name):
    if not sys.stdout.isatty(): return ""
    return {
        "reset":"\033[0m","muted":"\033[38;5;245m",
        "title":"\033[38;5;111m","num":"\033[38;5;81m","accent":"\033[38;5;45m",
    }.get(name,"")
def _vislen(s): return len(_ANSI_RE.sub("", s or ""))
def _pad_to(s,w): return s + (" " * max(0, w-_vislen(s)))
def _hr(w): return "─" * max(0,w)
def _menu_line(num,label,width_num=2,width_label=44):
    n=f"{_ansi('num')}{str(num).rjust(width_num)}{_ansi('reset')})"
    parts=textwrap.wrap(label,width_label) or [label]
    out=[f"{n}  {parts[0]}"]
    for ln in parts[1:]:
        out.append(" " * (width_num+3) + ln)
    return out
def panel(title, items):
    os.system("clear")
    cols=shutil.get_terminal_size(fallback=(100,28)).columns
    panel_w=min(90,max(60,cols-4))
    top="┌"+_hr(panel_w)+"┐"; sep="├"+_hr(panel_w)+"┤"; bot="└"+_hr(panel_w)+"┘"
    print(top)
    print("│ " + _pad_to(f"{_ansi('title')}{title}{_ansi('reset')}", panel_w-1) + "│")
    print(sep)
    width_num = 2 if max(len(k) for k,_,_ in items) <= 2 else 3
    width_label = panel_w - (width_num + 5)
    for k, label, _ in items:
        for ln in _menu_line(k,label,width_num,width_label):
            print("│ " + _pad_to(ln, panel_w-1) + "│")
    print(sep)
    tip=f"{_ansi('muted')}Ctrl+C: batal/ulang · Enter kosong: ulang menu{_ansi('reset')}"
    print("│ " + _pad_to(tip, panel_w-1) + "│")
    print(bot)

# ================== Data helpers ==================
def fetch_users_json(filters=None):
    args=["list","--limit","200","--max-offset","5000","--json"]
    if filters: args += filters
    r=run_lvmz(args, capture=True)
    if r.returncode!=0:
        print(r.stderr or r.stdout); return []
    try:
        return json.loads(r.stdout or "[]")
    except Exception:
        return []

def picker_user(allow_manual=True, filters=None):
    rows=fetch_users_json(filters)
    users=[]
    for u in rows:
        uname=u.get("username") or "-"
        st=(u.get("status") or "-").capitalize()
        used=human_bytes(u.get("used_traffic") or 0)
        limv=int(u.get("data_limit") or 0)
        lim="Unlimited" if limv==0 else human_bytes(limv)
        users.append((uname,st,used,lim))
    if not users:
        print("(tidak ada user)")
        if allow_manual:
            u=safe_input("Ketik username (kosong=batal): ")
            return (u or "").strip() or None
        return None
    print("\n=== USERS ===")
    for i,(u,st,used,lim) in enumerate(users,1):
        print(f"{i:2d}. {u:20s} {st:9s} used:{used:>10s} limit:{lim:>10s}")
    if allow_manual: print(" M. ketik username manual")
    print(" X. batal")
    while True:
        ch=safe_input("Pilih nomor/M/X: ")
        if ch is None or not ch.strip(): return None
        ch=ch.strip().lower()
        if ch=="x": return None
        if allow_manual and ch=="m":
            u=safe_input("Username: "); return (u or "").strip() or None
        if ch.isdigit():
            idx=int(ch)
            if 1<=idx<=len(users): return users[idx-1][0]
        print("Pilihan tidak valid.")

# ================== Pilihan angka ==================
def choose(title, options, default_idx=None):
    """
    options: list of (label, value) ; return chosen value
    """
    print(title)
    for i,(label,_) in enumerate(options,1):
        print(f" {i}) {label}")
    prompt = f"Pilih [1-{len(options)}]"
    if default_idx is not None:
        prompt += f" (default {default_idx})"
    prompt += ": "
    while True:
        ans=safe_input(prompt)
        if ans is None: return None
        ans=ans.strip()
        if ans=="" and default_idx is not None:
            return options[default_idx-1][1]
        if ans.isdigit():
            idx=int(ans)
            if 1<=idx<=len(options):
                return options[idx-1][1]
        print("Pilihan tidak valid.")

# ================== Ringkasan ADD ==================
def build_add_summary(proto: str, plugin: str, resp: dict, username: str, days_input: Optional[str]) -> str:
    import urllib.parse, time as _time
    domain = read_domain()

    # Durasi
    durasi = (f"{days_input} hari" if (days_input and days_input.isdigit()) else "AlwaysON")

    # Expired (UTC) + sisa hari
    exp_ts = int(resp.get("expire") or 0)
    if exp_ts > 0:
        exp_str = ts_to_utc(exp_ts)  # "YYYY-mm-dd HH:MM:SSZ"
        remain_days = max(0, (exp_ts - int(_time.time())) // 86400)
        expired_line = f"{exp_str}  ({remain_days}d)"
    else:
        expired_line = "AlwaysON"

    # Created at
    created_at = resp.get("created_at") or resp.get("createdAt")
    def fmt_created(a):
        if not a:
            return "-"
        try:
            dt = datetime.fromisoformat(a.replace("Z","+00:00"))
            return dt.strftime("%Y-%m-%d %H:%M:%S")
        except Exception:
            return str(a)
    created_line = fmt_created(created_at)

    # Subscription (absolute URL)
    sub_rel = resp.get("subscription_url") or ""
    sub_abs = f"https://{domain}{sub_rel}" if (sub_rel.startswith("/")) else (sub_rel or f"https://{domain}/sub")

    # kredensial (UUID/password)
    proxies = resp.get("proxies") or {}
    cred_id = "-"
    if proto == "vmess":
        cred_id = (proxies.get("vmess") or {}).get("id", "-")
    elif proto == "vless":
        cred_id = (proxies.get("vless") or {}).get("id", "-")
    elif proto == "trojan":
        cred_id = (proxies.get("trojan") or {}).get("password", "-")

    # Transport label
    transport = plugin

    # Ports
    if transport in ("ws", "hu"):
        ports_line = "TLS 443, nTLS 80"
    elif transport in ("grpc", "reality", "tcp"):
        ports_line = "TLS 443"
    else:
        ports_line = "TLS 443, nTLS 80"

    # Ambil links dari API
    links = resp.get("links") or resp.get("uris") or []

    # ---- Helpers untuk VMESS ----
    def _vmess_decode(uri: str):
        """vmess://<b64> → dict JSON (atau None)."""
        try:
            if not uri.startswith("vmess://"):
                return None
            b64 = uri[8:].strip()
            b64 += "=" * ((4 - len(b64) % 4) % 4)  # padding
            raw = base64.b64decode(b64).decode("utf-8", "ignore")
            return json.loads(raw)
        except Exception:
            return None

    def _vmess_is_tls(obj: dict) -> bool:
        tls_val = str(obj.get("tls") or obj.get("security") or "").lower()
        return tls_val in ("tls", "xtls")

    def _vmess_path(obj: dict, transport_hint: str) -> Optional[str]:
        net = (obj.get("net") or obj.get("type") or "").lower()
        t = (transport_hint or net).lower()
        # root fields
        if t in ("ws", "hu", "httpupgrade"):
            p = obj.get("path") or obj.get("ws-path") or obj.get("http-path")
            if p: return p
        if t == "grpc":
            sv = (obj.get("serviceName") or obj.get("grpc-serviceName") or
                  (obj.get("grpcSettings") or {}).get("serviceName"))
            if sv: return sv
        # nested ws
        ws = obj.get("wsSettings") or obj.get("ws-opts") or {}
        if t in ("ws", "hu") and isinstance(ws, dict):
            p = ws.get("path")
            if p: return p
        # nested grpc
        g = obj.get("grpcSettings") or obj.get("grpc-opts") or {}
        if t == "grpc" and isinstance(g, dict):
            sv = g.get("serviceName")
            if sv: return sv
        return None

    # ---- Ekstrak PATH ----
    path_line = "-"
    if links:
        if proto in ("vless", "trojan"):
            # parse dari query (?path= / serviceName= )
            def parse_path_from_uri(uri: str) -> Optional[str]:
                try:
                    if uri.startswith("vless://") or uri.startswith("trojan://"):
                        qpos = uri.find("?")
                        if qpos == -1:
                            return None
                        query = uri[qpos+1:]
                        if "#" in query:
                            query = query.split("#",1)[0]
                        params = urllib.parse.parse_qs(query, keep_blank_values=True)
                        if transport == "ws":
                            p = params.get("path", [""])[0]
                            return urllib.parse.unquote(p) if p else None
                        if transport == "grpc":
                            sv = params.get("serviceName", [""])[0]
                            return sv or None
                        if transport == "hu":
                            p = params.get("path", [""])[0]
                            return urllib.parse.unquote(p) if p else None
                        return None
                    return None
                except Exception:
                    return None
            for uri in links:
                got = parse_path_from_uri(uri)
                if got:
                    path_line = got
                    break
        elif proto == "vmess":
            for uri in links:
                obj = _vmess_decode(uri)
                if obj:
                    got = _vmess_path(obj, transport)
                    if got:
                        path_line = got
                        break

    # fallback default bila belum ada path
    if path_line == "-" and transport == "ws":
        path_line = f"/{proto}"
    elif path_line == "-" and transport == "hu":
        path_line = f"/{proto}-http"
    elif path_line == "-" and transport == "grpc":
        path_line = f"{proto}-service"

    # ---- URI TLS / nTLS ----
    uri_tls = None
    uri_ntls = None
    if links:
        if proto in ("vless", "trojan"):
            def is_tls(uri: str) -> bool: return "security=tls" in uri
            for uri in links:
                if uri.startswith(("vless://","trojan://")) and is_tls(uri):
                    uri_tls = uri; break
            for uri in links:
                if uri.startswith(("vless://","trojan://")) and not is_tls(uri):
                    uri_ntls = uri; break
        elif proto == "vmess":
            for uri in links:
                obj = _vmess_decode(uri)
                if obj and _vmess_is_tls(obj):
                    uri_tls = uri; break
            for uri in links:
                obj = _vmess_decode(uri)
                if obj and not _vmess_is_tls(obj):
                    uri_ntls = uri; break

    # Banner
    H = "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
    title = f"⇱ {proto} {transport} Account Created ⇲".upper() if proto != "vmess" else f"⇱ {proto} {transport} Account Created ⇲"
    cred_label = "UUID" if proto in ("vmess","vless") else "Password"

    out = []
    out.append(H)
    out.append(f"           {title}")
    out.append(H)
    out.append(f"Username: {username}")
    out.append(f"Domain: {domain}")
    if cred_id != "-":
        out.append(f"{cred_label}: {cred_id}")
    out.append(f"Ports: {ports_line}")
    out.append(f"Durasi: {durasi}")
    out.append(f"Protocol: {proto}")
    out.append(f"Transport: {transport}")
    if path_line and path_line != "-":
        out.append(f"Path: {path_line}")
    out.append(f"Subscription: {sub_abs}")
    out.append(f"Expired: {expired_line}")
    out.append(f"Dibuat pada: {created_line}")
    out.append(H)
    if uri_tls:
        out.append(f"URI [TLS]: {uri_tls}")
    if uri_ntls:
        out.append(f"URI [nTLS]: {uri_ntls}")
    return "\n".join(out)

# ================== Actions ==================
def act_add():
    os.system("clear")
    print("Tambah user")

    u=safe_input("Username: ")
    if not u: print("batal."); return
    u=u.strip()

    proto = choose("Protocol:",
                   [("VMESS","vmess"),("VLESS","vless"),("TROJAN","trojan")],
                   default_idx=1)
    if proto is None: print("batal."); return

    # Pilihan plugin per-protocol (tanpa kombinasi ws+grpc+…)
    if proto == "vmess":
        plugin = choose("Plugin:",
                        [("All","all"),
                         ("WebSocket (ws)","ws"),
                         ("gRPC (grpc)","grpc"),
                         ("HTTP Upgrade (hu)","hu")],
                        default_idx=1)
    elif proto == "trojan":
        plugin = choose("Plugin:",
                        [("All","all"),
                         ("tcp","tcp"),
                         ("WebSocket (ws)","ws"),
                         ("gRPC (grpc)","grpc"),
                         ("HTTP Upgrade (hu)","hu")],
                        default_idx=1)
        if plugin == "tcp":  # backend belum support 'tcp' → treat as 'all'
            plugin = "all"
    else:  # vless
        plugin = choose("Plugin:",
                        [("All","all"),
                         ("WebSocket (ws)","ws"),
                         ("gRPC (grpc)","grpc"),
                         ("HTTP Upgrade (hu)","hu"),
                         ("Reality (reality)","reality")],
                        default_idx=1)
    if plugin is None: print("batal."); return

    days  = (safe_input("Masa aktif (hari, kosong=AlwaysON): ") or "").strip()
    quota = (safe_input("Quota (GB, kosong=Unlimited): ") or "").strip()
    maxd  = (safe_input("Max devices [3]: ") or "3").strip()
    note  = (safe_input("Catatan (opsional): ") or "").strip()

    reset = choose("Reset strategy:",
                   [("No reset","no_reset"),
                    ("Harian","day"),
                    ("Mingguan","week"),
                    ("Bulanan","month"),
                    ("Tahunan","year")],
                   default_idx=4)
    if reset is None: print("batal."); return

    args = ["add", u, "--proto", proto, "--reset", reset]
    if plugin: args += ["--plugin", plugin]
    if days:   args += ["--days", days]
    if quota:  args += ["--quota-gb", quota]
    if note:   args += ["--note", note]
    # PENTING: JANGAN --set-flow di ADD

    r = run_lvmz(args + ["--json"], capture=True)
    if r.returncode != 0:
        print(r.stderr or r.stdout)
        tg_send(f"❌ ADD gagal: {u}\n{(r.stderr or r.stdout)[:4000]}", html=False)
        return

    # set max devices via modify (meta)
    try:
        if maxd and maxd.isdigit():
            _ = run_lvmz(["modify", u, "--max-dev", maxd, "--quiet"])
            print(f"OK modified username={u} max-dev={maxd}")
    except Exception:
        pass

    # kalau VLESS → set flow via modify TANPA --proto agar tidak mengubah inbounds
    if proto == "vless":
        try:
            _ = run_lvmz(["modify", u, "--set-flow", "xtls-rprx-vision", "--quiet"])
            print(f"OK set flow vision for {u}")
        except Exception:
            pass

    # parse JSON dari ADD
    try:
        data = json.loads(r.stdout)
    except Exception:
        print(r.stdout or "(no output)")
        tg_send(f"✅ ADD (raw): {u}\n{(r.stdout or '')[:1000]}", html=False)
        pause()
        return

    # cetak ringkasan baru (banner + URI dari response)
    summary = build_add_summary(
        proto=proto,
        plugin=plugin,
        resp=data,
        username=u,
        days_input=(days or None),
    )
    print(summary)
    tg_send(tg_format_summary_html(summary), html=True)  # HTML monospace
    pause()

def act_modify():
    os.system("clear")
    print("Modify user (pilih apa yang mau diubah)")
    u = picker_user(True)
    if not u:
        print("batal."); return

    OPTIONS = [
        ("Protocol (vmess/vless/trojan)", "proto"),
        ("Plugin (all/ws/grpc/hu/reality)", "plugin"),
        ("Catatan (note)", "note"),
        ("Set expire (hari)", "days"),
        ("Set quota GB", "quota"),
        ("Set Unlimited quota (ya/tidak)", "unlimited"),
        ("Reset strategy (no_reset/day/week/month/year)", "reset"),
        ("Max devices", "maxd"),
        ("Regen ID/PASS (vmess/vless=ID, trojan=PASS)", "regen"),
        ("VLESS flow (kosong=xtls-rprx-vision)", "flow"),
    ]
    for i,(lbl,_) in enumerate(OPTIONS,1):
        print(f" {i}) {lbl}")
    print(" 0) Batal")

    sel = (safe_input("Pilih angka (pisahkan koma, mis. 1,2,8): ") or "").strip()
    if not sel or sel == "0":
        print("batal."); return
    picks = [p.strip() for p in sel.split(",") if p.strip().isdigit()]

    vals = {}
    for p in picks:
        i = int(p)
        if not (1 <= i <= len(OPTIONS)): continue
        key = OPTIONS[i-1][1]
        if key == "proto":
            vals["proto"] = (safe_input("Protocol (vmess/vless/trojan): ") or "").strip().lower()
        elif key == "plugin":
            vals["plugin"] = (safe_input("Plugin (all/ws/grpc/hu/reality): ") or "").strip().lower()
        elif key == "note":
            vals["note"] = (safe_input("Catatan: ") or "").strip()
        elif key == "days":
            vals["days"] = (safe_input("Expire (hari): ") or "").strip()
        elif key == "quota":
            vals["quota"] = (safe_input("Quota GB: ") or "").strip()
        elif key == "unlimited":
            yn = (safe_input("Unlimited? [y/N]: ") or "n").strip().lower()
            vals["unlimited"] = (yn in ("y","yes"))
        elif key == "reset":
            vals["reset"] = (safe_input("Reset (no_reset/day/week/month/year): ") or "").strip().lower()
        elif key == "maxd":
            vals["maxd"] = (safe_input("Max devices: ") or "").strip()
        elif key == "regen":
            ptmp = (vals.get("proto") or (safe_input("Target proto regen (vmess/vless/trojan): ") or "")).strip().lower()
            if ptmp in ("vmess","vless"):
                vals["regen"] = "--regen-id"; vals["proto"] = ptmp or vals.get("proto")
            elif ptmp == "trojan":
                vals["regen"] = "--regen-pass"; vals["proto"] = ptmp or vals.get("proto")
        elif key == "flow":
            vals["flow"] = (safe_input("VLESS flow (kosong=xtls-rprx-vision): ") or "xtls-rprx-vision").strip()

    args = ["modify", u]
    if vals.get("proto"):   args += ["--proto", vals["proto"]]
    if vals.get("plugin"):  args += ["--plugin", vals["plugin"]]
    if vals.get("note"):    args += ["--note", vals["note"]]
    if vals.get("days"):    args += ["--days", vals["days"]]
    if vals.get("reset"):   args += ["--reset", vals["reset"]]
    if vals.get("quota"):   args += ["--quota-gb", vals["quota"]]
    if vals.get("maxd"):    args += ["--max-dev", vals["maxd"]]
    if vals.get("unlimited"): args += ["--unlimited"]
    if vals.get("regen"):   args += [vals["regen"]]
    if vals.get("flow"):    args += ["--set-flow", vals["flow"]]

    rc = run_lvmz(args)
    if rc==0:  tg_send(f"✏️ MODIFY: {u} (proto={vals.get('proto','-')} plugin={vals.get('plugin','-')})")
    else:      tg_send(f"❌ MODIFY gagal: {u}")
    pause()

def act_enable():
    os.system("clear")
    u=picker_user(True, ["--status","disabled"])
    if not u: print("batal."); return
    _ = run_lvmz(["enable", u])
    pause()  # no telegram

def act_disable():
    os.system("clear")
    u=picker_user(True, ["--status","active"])
    if not u: print("batal."); return
    _ = run_lvmz(["disable", u])
    pause()  # no telegram

def act_delete():
    os.system("clear")
    u=picker_user(True)
    if not u: print("batal."); return
    yn=(safe_input(f"Yakin hapus '{u}'? [y/N]: ") or "n").strip().lower()
    if yn not in ("y","yes"): print("batal."); return
    rc=run_lvmz(["delete", u])
    if rc==0:
        tg_send(f"🗑️ DELETE: {u}")
    else:
        tg_send(f"❌ DELETE gagal: {u}")
    pause()

def act_purge_all():
    os.system("clear")
    print("HAPUS SEMUA USER!")
    yn=(safe_input("Ketik 'YA' untuk lanjut: ") or "").strip().upper()
    if yn!="YA": print("batal."); return
    _ = run_lvmz(["purge-all","--force"])
    pause()  # no telegram

def act_purge_expired():
    os.system("clear")
    print("Hapus user EXPIRED")
    yn=(safe_input("Ketik 'YA' untuk lanjut: ") or "").strip().upper()
    if yn!="YA": print("batal."); return
    _ = run_lvmz(["purge-expired","--force"])
    pause()  # no telegram

def act_list():
    os.system("clear")
    contains=(safe_input("Filter username mengandung (kosong=semua): ") or "").strip()
    proto=(safe_input("Filter proto (vmess/vless/trojan/shadowsocks/all) [all]: ") or "all").strip().lower()
    status=(safe_input("Filter status (active/disabled/on_hold/expired/kosong=semua): ") or "").strip()
    extra=[]
    if contains: extra+=["--contains", contains]
    if status:   extra+=["--status", status]
    if proto!="all": extra+=["--proto", proto]
    run_lvmz(["list", *extra])
    pause()  # no telegram

def act_status():
    os.system("clear")
    u=picker_user(True)
    if not u: print("batal."); return
    run_lvmz(["status", u])
    pause()  # no telegram

def act_ips():
    os.system("clear")
    print("Cek IP user (berdasar access.log)")
    proto=(safe_input("Proto (vmess/vless/trojan/shadowsocks/all) [all]: ") or "all").strip().lower()
    only_active=(safe_input("Hanya user aktif? [Y/n]: ") or "y").strip().lower() in ("y","yes","")
    asn=(safe_input("Tampilkan ASN/ISP? [y/N]: ") or "n").strip().lower() in ("y","yes")

    args=["ips","--proto",proto]
    if only_active: args.append("--only-active")
    if asn: args.append("--asn")

    run_lvmz(args)
    pause()  # no telegram

def act_extend():
    os.system("clear")
    u=picker_user(True)
    if not u: print("batal."); return
    d=(safe_input("Tambah hari (0=skip) [30]: ") or "30").strip()
    h=(safe_input("Tambah jam (0=skip) [0]: ") or "0").strip()
    args=["extend", u]
    if d and d!="0": args+=["--days", d]
    if h and h!="0": args+=["--hours", h]
    rc=run_lvmz(args)
    if rc==0:
        tg_send(f"⏳ EXTEND: {u} (+{d}d {h}h)")
    else:
        tg_send(f"❌ EXTEND gagal: {u}")
    pause()

def act_reset_usage():
    os.system("clear")
    print("Reset usage")
    print("1) Reset satu user")
    print("2) Reset SEMUA user")
    print("0) Batal")
    ch=(safe_input("Pilih: ") or "").strip()
    if ch=="1":
        u=picker_user(True)
        if not u: print("batal."); return
        rc=run_lvmz(["reset-quota", u])
        if rc==0:
            tg_send(f"♻️ RESET usage: {u}")
        else:
            tg_send(f"❌ RESET usage gagal: {u}")
    elif ch=="2":
        yn=(safe_input("Ketik 'YA' untuk reset semua: ") or "").strip().upper()
        if yn!="YA": print("batal."); return
        rows=fetch_users_json()
        ok=fail=0
        for row in rows:
            name=row.get("username")
            if not name: continue
            rc=run_lvmz(["reset-quota", name])
            ok += 1 if rc==0 else 0
            fail += 0 if rc==0 else 1
        print(f"Selesai. OK={ok} FAIL={fail}")
        tg_send(f"♻️ RESET usage ALL selesai: OK={ok} FAIL={fail}")
    else:
        print("batal.")
    pause()

def act_fix_ssl():
    os.system("clear")
    domain = read_domain()
    print("Fix SSL (acme.sh, standalone)")
    print(f"Domain: {domain}")
    email = (safe_input("Masukkan Email: ") or "").strip()
    if not email:
        print("batal."); return

    compose_dir = "/opt/marzban"
    crt = "/var/lib/marzban/xray.crt"
    key = "/var/lib/marzban/xray.key"

    def run(cmd):
        return subprocess.run(["bash","-lc", cmd], check=True)

    try:
        if os.path.isdir(compose_dir):
            subprocess.run(["bash","-lc", f"cd {compose_dir} && docker compose down"], check=False)

        run(f"/root/.acme.sh/acme.sh --server letsencrypt --register-account -m {shlex.quote(email)}")
        run(f"/root/.acme.sh/acme.sh --server letsencrypt --issue -d {shlex.quote(domain)} --standalone -k ec-256 --force --debug")
        run(f"~/.acme.sh/acme.sh --installcert -d {shlex.quote(domain)} --fullchainpath {shlex.quote(crt)} --keypath {shlex.quote(key)} --ecc")

        if os.path.isdir(compose_dir):
            subprocess.run(["bash","-lc", f"cd {compose_dir} && docker compose up -d"], check=False)

        print("\n=== CERT INSTALLED ===")
        try:
            print(Path(crt).read_text().splitlines()[0])
            print(Path(key).read_text().splitlines()[0])
        except Exception:
            pass
        print("======================")
    except subprocess.CalledProcessError as e:
        print("Gagal proses SSL:", e)
    pause()

def act_seeroute():
    """
    Tampilkan daftar outbound (tag, protocol) + negara endpoint + daftar domain rules.
    Sumber data: /var/lib/marzban/xray_config.json
    Negara di-cache di /tmp/seeroute_cache (format: IP=CC).
    """
    os.system("clear")
    cfg = "/var/lib/marzban/xray_config.json"
    cache_file = "/tmp/seeroute_cache"

    if not os.path.exists(cfg):
        print(f"Config {cfg} tidak ditemukan.")
        pause()
        return

    # baca config
    try:
        conf = json.load(open(cfg, "r", encoding="utf-8"))
    except Exception as e:
        print(f"Gagal baca JSON: {e}")
        pause()
        return

    # filter outbound tag (opsional)
    flt = (safe_input("Filter outboundTag (kosong=semua): ") or "").strip()

    # muat cache negara
    cache = {}
    try:
        if os.path.exists(cache_file):
            for ln in open(cache_file, "r", encoding="utf-8"):
                ln = ln.strip()
                if not ln or "=" not in ln: 
                    continue
                ip, cc = ln.split("=", 1)
                cache[ip.strip()] = cc.strip()
    except Exception:
        cache = {}

    ip_re = re.compile(r"^\d{1,3}(?:\.\d{1,3}){3}$")

    def resolve_ip(host: str) -> str:
        if not host:
            return ""
        if ip_re.match(host):
            return host
        # coba socket dulu
        try:
            return socket.gethostbyname(host)
        except Exception:
            pass
        # fallback getent
        try:
            r = subprocess.run(["getent", "hosts", host], text=True, capture_output=True, timeout=5)
            if r.returncode == 0 and r.stdout.strip():
                return r.stdout.strip().split()[0]
        except Exception:
            pass
        # fallback dig
        try:
            r = subprocess.run(["dig", "+short", host, "A"], text=True, capture_output=True, timeout=5)
            if r.returncode == 0 and r.stdout.strip():
                return r.stdout.strip().splitlines()[0]
        except Exception:
            pass
        return ""

    def lookup_country(ip: str) -> str:
        if not ip:
            return "-"
        if ip in cache:
            return cache[ip]
        # ipinfo simple, hanya kode negara (2 huruf)
        try:
            with urllib.request.urlopen(f"https://ipinfo.io/{ip}/country", timeout=5) as resp:
                cc = (resp.read().decode("utf-8", "ignore") or "").strip() or "Unknown"
        except Exception:
            cc = "Unknown"
        # simpan cache
        try:
            cache[ip] = cc
            with open(cache_file, "a", encoding="utf-8") as f:
                f.write(f"{ip}={cc}\n")
        except Exception:
            pass
        return cc

    # ambil daftar rules utk pemetaan domain
    rules = (conf.get("routing") or {}).get("rules") or []

    def domains_for_tag(tag: str):
        out = []
        for r in rules:
            try:
                if r.get("outboundTag") == tag:
                    ds = r.get("domain") or []
                    # kadang string tunggal
                    if isinstance(ds, str):
                        out.append(ds)
                    elif isinstance(ds, list):
                        out.extend([str(x) for x in ds])
            except Exception:
                continue
        return out

    # header
    print()
    print("OutboundTag       Protocol    Country")
    print("==============================================")

    outbounds = conf.get("outbounds") or []
    for ob in outbounds:
        try:
            tag = ob.get("tag") or "-"
            proto = ob.get("protocol") or "-"
            if tag in ("direct", "block", "dns-out"):
                continue
            if flt and flt not in tag:
                continue

            addr = ""
            st = ob.get("settings") or {}
            # vmess/vless/trojan common: settings.servers[0].address
            try:
                addr = (st.get("servers") or [{}])[0].get("address") or ""
            except Exception:
                addr = ""
            # reality/tujuan peer (trojan/grpc reality): settings.peers[0].endpoint (host:port)
            if not addr:
                try:
                    ep = (st.get("peers") or [{}])[0].get("endpoint") or ""
                    if ":" in ep:
                        addr = ep.split(":", 1)[0]
                except Exception:
                    pass

            ip = resolve_ip(addr) if addr else ""
            cc = lookup_country(ip) if ip else "-"

            print(f"{tag:<16} {proto:<10} {cc}")

            doms = domains_for_tag(tag)
            if doms:
                for d in doms:
                    print(f"    -> {d}")
            else:
                print("    (no domains)")
            print("")
        except KeyboardInterrupt:
            raise
        except Exception as e:
            print(f"(skip satu outbound: {e})")
            print("")

    pause()

# ================== Main menu ==================
def main():
    req_root()
    if not os.path.exists(LVMZ):
        print(f"Error: {LVMZ} tidak ditemukan."); sys.exit(1)

    items=[
        ("1","Tambah user",                  lambda: act_add()),
        ("2","Modify user",                  lambda: act_modify()),
        ("3","Enable user",                  lambda: act_enable()),
        ("4","Disable user",                 lambda: act_disable()),
        ("5","Delete user",                  lambda: act_delete()),
        ("6","Hapus SEMUA user",             lambda: act_purge_all()),
        ("7","Hapus user expired",           lambda: act_purge_expired()),
        ("8","List users",                   lambda: act_list()),
        ("9","Status user",                  lambda: act_status()),
        ("10","Cek IP user",                 lambda: act_ips()),
        ("11","Extend masa aktif",           lambda: act_extend()),
        ("12","Reset usage",                 lambda: act_reset_usage()),
        ("13","Fix SSL (acme.sh)",           lambda: act_fix_ssl()),
        ("14","See Route (country)",         lambda: act_seeroute()),
        ("0","Keluar",                       None),
    ]

    while True:
        panel("XRAY/Marzban Menu", items)
        ch=safe_input(f"{_ansi('accent')}Pilih{_ansi('reset')}: ")
        if ch is None:  # Ctrl+C → ulang
            continue
        ch=ch.strip()
        matched=False
        for k,_,fn in items:
            if ch==k:
                matched=True
                if k=="0":
                    print("bye."); return
                try:
                    fn()
                except KeyboardInterrupt:
                    print("\n(dibatalkan)")
                break
        if not matched and ch!="":
            print("Pilihan tidak dikenal."); time.sleep(0.8)

if __name__ == "__main__":
    main()
