#!/usr/bin/env python3
# LingVPN Marzban Manager – add/delete/list/modify (PUT modify) + minimal stdout + save to /var/lib/marzban
# deps: apt install -y python3-requests
import argparse, json, os, re, sys, uuid as uuidlib, pathlib, time, math, ipaddress, subprocess
from datetime import datetime, timedelta, timezone
IP_RE = re.compile(r'(\d{1,3}(?:\.\d{1,3}){3})')

try:
    import requests
except ImportError:
    print("GAGAL: module 'requests' belum terpasang (apt install -y python3-requests)", file=sys.stderr)
    sys.exit(1)

ASIA_JAKARTA = timezone(timedelta(hours=7))
USERNAME_RE = re.compile(r"^[a-zA-Z0-9_]{3,32}$")
DATA_DIR = "/var/lib/marzban"
META_DIR = f"{DATA_DIR}/meta"
LOCK_DIR = f"{DATA_DIR}/locks"
MAXDEV_PATH = f"{META_DIR}/max_devices.json"

# ---------- io helpers ----------
def collapse_label(ip, mode):
    """
    mode '/24' -> 'a.b.c.xx' untuk IPv4; 'ipv6' nanti bisa ditambah.
    mode 'none' -> kembalikan ip asli.
    """
    if mode == "none":
        return ip
    if mode == "/24":
        # hanya IPv4: ambil 3 oktet pertama
        parts = ip.split(".")
        if len(parts) == 4:
            return f"{parts[0]}.{parts[1]}.{parts[2]}.xx"
        # kalau bukan IPv4, kembalikan apa adanya
        return ip
    return ip

def is_private_ip(ip):
    try:
        return ipaddress.ip_address(ip).is_private
    except Exception:
        return True

def to_wib_str(iso_str):
    if not iso_str or iso_str == "null":
        return "null"
    try:
        # parse ISO dari API → WIB
        dt = datetime.fromisoformat(iso_str.replace("Z","+00:00"))
        return dt.astimezone(ASIA_JAKARTA).strftime("%Y-%m-%d %H:%M:%S")
    except Exception:
        return iso_str

def fetch_users_filtered(proto=None, contains=None, only_active=False, limit=1000):
    base = api_base()
    out = []
    offset = 0
    while True:
        data = http_json("get", f"{base}/users", params={"limit": limit, "offset": offset}, headers=bearer())
        users = data.get("users") or []
        if not users: break
        for u in users:
            uname = u.get("username") or ""
            if contains and contains not in uname:
                continue
            if proto and proto not in (list((u.get("proxies") or {}).keys())) and proto != "all":
                continue
            if only_active and (u.get("status") or "").lower() != "active":
                continue
            out.append(u)
        if len(users) < limit: break
        offset += limit
    return out

def build_username_regex(usernames):
    # \b untuk aman; username kita [\w_] jadi aman di regex
    if not usernames:
        return None
    escaped = [re.escape(u) for u in usernames]
    pattern = r'\b(?:' + "|".join(escaped) + r')\b'
    return re.compile(pattern)

def ipinfo_org(ip):
    try:
        token = os.environ.get("IPINFO_TOKEN", "")
        url = f"http://ipinfo.io/{ip}/org"
        headers = {}
        if token:
            headers["Authorization"] = f"Bearer {token}"
        r = requests.get(url, headers=headers, timeout=6)
        if not r.ok: 
            return None
        return r.text.strip().strip('"')
    except Exception:
        return None

def confirm_or_exit(warning_text, force=False):
    if force:
        return
    print(warning_text)
    ans = input("Ketik 'YES' untuk melanjutkan: ").strip()
    if ans != "YES":
        print("Dibatalkan.")
        sys.exit(1)

def read_token(path="/root/token.json"):
    try:
        with open(path, "r") as f:
            tok = json.load(f)
            return tok.get("access_token") or tok.get("token") or ""
    except Exception:
        return ""

def get_marzban_port(env_file="/opt/marzban/.env"):
    try:
        with open(env_file, "r") as f:
            for line in f:
                if line.strip().startswith("UVICORN_PORT"):
                    m = re.search(r"=\s*['\"]?(\d+)['\"]?", line)
                    if m:
                        return m.group(1)
    except Exception:
        pass
    return "7879"

def api_base():
    host = os.environ.get("API_HOST", "127.0.0.1")
    port = os.environ.get("API_PORT", get_marzban_port())
    return f"http://{host}:{port}/api"

def bearer():
    token = os.environ.get("API_TOKEN") or read_token()
    if not token:
        print("GAGAL: token API tidak ditemukan (/root/token.json atau env API_TOKEN)", file=sys.stderr)
        sys.exit(1)
    return {"Authorization": f"Bearer {token}"}

def http_raw(method, url, **kw):
    headers = kw.pop("headers", {})
    headers["accept"] = "application/json"
    if method in ("post","put","patch"):
        headers["Content-Type"] = "application/json"
    try:
        r = requests.request(method.upper(), url, headers=headers, **kw, timeout=30)
    except requests.RequestException as e:
        print(f"GAGAL: koneksi API: {e}", file=sys.stderr); sys.exit(1)
    return r

def http_json(method, url, **kw):
    r = http_raw(method, url, **kw)
    if not r.ok:
        msg = r.text.strip().replace("\n", " ")
        if len(msg) > 300: msg = msg[:300] + "…"
        print(f"GAGAL: HTTP {r.status_code}: {msg}", file=sys.stderr)
        sys.exit(1)
    return r.json() if r.text else {}

# ---------- dirs ----------
def ensure_dirs():
    for d in (f"{DATA_DIR}/users", META_DIR, LOCK_DIR):
        pathlib.Path(d).mkdir(parents=True, exist_ok=True)

# ---------- meta: max-dev ----------
def load_maxdev():
    try:
        with open(MAXDEV_PATH, "r", encoding="utf-8") as f:
            return json.load(f)
    except Exception:
        return {}

def save_maxdev(mapping):
    pathlib.Path(META_DIR).mkdir(parents=True, exist_ok=True)
    with open(MAXDEV_PATH, "w", encoding="utf-8") as f:
        json.dump(mapping, f, ensure_ascii=False, indent=2)

def set_user_maxdev(username, n):
    m = load_maxdev()
    if n is None:
        if username in m:
            del m[username]
    else:
        m[username] = int(n)
    save_maxdev(m)

def get_user_maxdev(username, default=None):
    m = load_maxdev()
    return m.get(username, default)

# ---------- time & size helpers ----------
def now_utc():
    return datetime.now(timezone.utc)

def iso_in(days=14):
    return (datetime.now(ASIA_JAKARTA) + timedelta(days=days)).astimezone(timezone.utc).replace(microsecond=0).isoformat()

def expire_ts(days):
    if days is None or days <= 0:
        return None
    return int((now_utc() + timedelta(days=days)).timestamp())

def human_bytes(n):
    try:
        n = float(n or 0)
    except Exception:
        n = 0.0
    units = ["B","KB","MB","GB","TB","PB"]
    i = 0
    while n >= 1024 and i < len(units)-1:
        n /= 1024.0
        i += 1
    if i == 0:
        return f"{int(n)} B"
    return f"{n:.2f} {units[i]}"

def gb_2dec_from_bytes(n):
    if not n:
        return "0.00 GB"
    return f"{(n/1024/1024/1024):.2f} GB"

def map_reset(name):
    return {
        None: "-",
        "no_reset": "-",
        "day": "Daily",
        "week": "Weekly",
        "month": "Monthly",
        "year": "Yearly",
    }.get(name, name)

def rel_days_str(ts):
    try:
        ts = int(ts)
    except Exception:
        return "-"
    remain = ts - int(now_utc().timestamp())
    if remain <= 0:
        return "(0d)"
    days = math.ceil(remain / 86400)
    return f"({days}d)"

# ---------- inbounds presets ----------
INBOUND_PRESETS = {
    "vmess": {
        "ws":   ["VMESS_WS","VMESS_WS_ANTIADS","VMESS_WS_ANTIPORN"],
        "grpc": ["VMESS_GRPC"],
        "hu":   ["VMESS_HTTPUPGRADE","VMESS_HU_ANTIADS","VMESS_HU_ANTIPORN"],
    },
    "vless": {
        "ws":   ["VLESS_WS","VLESS_WS_ANTIADS","VLESS_WS_ANTIPORN"],
        "grpc": ["VLESS_GRPC"],
        "hu":   ["VLESS_HTTPUPGRADE","VLESS_HU_ANTIADS","VLESS_HU_ANTIPORN"],
        "reality": ["VLESS_REALITY_FALLBACK","VLESS_REALITY_GRPC"],
    },
    "trojan": {
        "tcp":  ["TROJAN_TCP"],
        "ws":   ["TROJAN_WS","TROJAN_WS_ANTIADS","TROJAN_WS_ANTIPORN"],
        "grpc": ["TROJAN_GRPC"],
        "hu":   ["TROJAN_HTTPUPGRADE","TROJAN_HU_ANTIADS","TROJAN_HU_ANTIPORN"],
    },
}

def build_inbounds(proto, plugins):
    chosen = set()
    presets = INBOUND_PRESETS.get(proto, {})
    if "all" in plugins:
        for v in presets.values():
            chosen.update(v)
    else:
        for p in plugins:
            chosen.update(presets.get(p, []))
    return {proto: sorted(chosen)} if chosen else {}

# ---------- proxies builders ----------
def build_proxies(proto, user_uuid=None, vless_flow=None, trojan_pass=None):
    if proto == "vmess":
        return {"vmess": {"id": user_uuid or str(uuidlib.uuid4())}}
    if proto == "vless":
        p = {"id": user_uuid or str(uuidlib.uuid4())}
        if vless_flow:
            p["flow"] = vless_flow  # xtls-rprx-vision
        return {"vless": p}
    if proto == "trojan":
        return {"trojan": {"password": trojan_pass or (user_uuid or str(uuidlib.uuid4()))}}
    print("GAGAL: proto tidak didukung", file=sys.stderr); sys.exit(1)

# ---------- storage ----------
def safe_makedirs(p, mode=0o755):
    pathlib.Path(p).mkdir(parents=True, exist_ok=True)
    try: os.chmod(p, mode)
    except Exception: pass

def save_response(username, proto, payload, out_dir=DATA_DIR):
    ts = int(time.time())
    base = pathlib.Path(out_dir)
    user_dir = base / "users" / proto
    safe_makedirs(user_dir)

    record = {
        "saved_at": ts,
        "username": username,
        "proto": proto,
        "payload": payload
    }

    snap_path = user_dir / f"{username}.json"
    with open(snap_path, "w", encoding="utf-8") as f:
        json.dump(record, f, ensure_ascii=False, indent=2)

    index_path = base / "users" / "users.jsonl"
    safe_makedirs(index_path.parent)
    with open(index_path, "a", encoding="utf-8") as f:
        f.write(json.dumps(record, ensure_ascii=False) + "\n")

def remove_snapshot(username, out_dir=DATA_DIR):
    base = pathlib.Path(out_dir) / "users"
    if base.exists():
        for proto_dir in base.iterdir():
            p = proto_dir / f"{username}.json"
            try:
                if p.exists():
                    p.unlink()
            except Exception:
                pass

# ---------- user helpers ----------
def check_username_unique(base, uname):
    data = http_json("get", f"{base}/users", params={"username": uname}, headers=bearer())
    return int(data.get("total", 0)) == 0

def get_user(base, username):
    return http_json("get", f"{base}/user/{username}", headers=bearer())

# ---------- commands ----------
def cmd_enable(args):
    base = api_base()
    current = get_user(base, args.username)
    body = sanitize_user_for_put(current)
    body["status"] = "active"
    # jangan ubah field lain
    resp = http_json("put", f"{base}/user/{args.username}", headers=bearer(), data=json.dumps(body))
    save_response(args.username, list((resp.get("proxies") or {}).keys() or ["vmess"])[0], resp, "/var/lib/marzban")
    print("OK")

def cmd_disable(args):
    base = api_base()
    current = get_user(base, args.username)
    body = sanitize_user_for_put(current)
    body["status"] = "disabled"
    body["on_hold_timeout"] = None
    body["on_hold_expire_duration"] = 0
    resp = http_json("put", f"{base}/user/{args.username}", headers=bearer(), data=json.dumps(body))
    save_response(args.username, list((resp.get("proxies") or {}).keys() or ["vmess"])[0], resp, "/var/lib/marzban")
    print("OK")

def cmd_add(args):
    ensure_dirs()
    base = api_base()
    if not USERNAME_RE.match(args.username):
        print("GAGAL: username harus 3–32 & [a-zA-Z0-9_]", file=sys.stderr); sys.exit(1)
    if not check_username_unique(base, args.username):
        print(f"GAGAL: username {args.username} sudah ada", file=sys.stderr); sys.exit(1)

    if args.unlimited:
        data_limit = 0
    else:
        if args.quota_gb is None or args.quota_gb <= 0:
            print("GAGAL: --quota-gb harus > 0 atau pakai --unlimited", file=sys.stderr); sys.exit(1)
        data_limit = int(args.quota_gb * (1024**3))

    expire = expire_ts(args.days) if args.days is not None else None

    want_flow = None
    if args.proto == "vless" and ("reality" in args.plugin or "all" in args.plugin):
        want_flow = "xtls-rprx-vision"

    user_uuid = str(uuidlib.uuid4())
    proxies = build_proxies(args.proto, user_uuid=user_uuid, vless_flow=want_flow)
    inbounds = build_inbounds(args.proto, args.plugin)

    payload = {
        "username": args.username,
        "proxies": proxies,
        "inbounds": inbounds,
        "data_limit": data_limit,
        "data_limit_reset_strategy": args.reset,
        "note": args.note or "",
    }

    if args.on_hold:
        payload["status"] = "on_hold"
        payload["on_hold_timeout"] = iso_in(args.hold_days)
        payload["on_hold_expire_duration"] = int((args.days or 0) * 24 * 3600) if args.days else 0
    else:
        payload["status"] = "active"
        payload["expire"] = expire or 0

    resp = http_json("post", f"{base}/user", headers=bearer(), data=json.dumps(payload))
    save_response(args.username, args.proto, resp, args.out_dir)

    # simpan max-dev (metadata lokal)
    if args.max_dev is not None:
        set_user_maxdev(args.username, args.max_dev)

    if args.json:
        print(json.dumps(resp, ensure_ascii=False))
    else:
        if args.quiet:
            print("OK")
        else:
            sub = resp.get("subscription_url") or "-"
            md = args.max_dev if args.max_dev is not None else "-"
            print(f"OK username={args.username} proto={args.proto} plugins={','.join(args.plugin)} flow={want_flow or '-'} max-dev={md} sub={sub}")

def cmd_delete(args):
    ensure_dirs()
    base = api_base()
    r = http_raw("get", f"{base}/user/{args.username}", headers=bearer())
    if r.status_code == 404:
        print(f"GAGAL: username {args.username} tidak ditemukan", file=sys.stderr); sys.exit(1)
    r = http_raw("delete", f"{base}/user/{args.username}", headers=bearer())
    if not r.ok:
        msg = r.text.strip().replace("\n", " ")
        if len(msg) > 300: msg = msg[:300] + "…"
        print(f"GAGAL: HTTP {r.status_code}: {msg}", file=sys.stderr); sys.exit(1)
    remove_snapshot(args.username, args.out_dir)
    # hapus max-dev meta
    set_user_maxdev(args.username, None)
    if args.quiet:
        print("OK")
    else:
        print(f"OK deleted username={args.username}")

def cmd_list(args):
    ensure_dirs()
    base = api_base()
    limit = args.limit
    offset = 0
    printed = 0
    rows = []

    while True:
        params = {"limit": limit, "offset": offset}
        if args.status:
            params["status"] = args.status
        if args.username:
            params["username"] = args.username
        r = http_json("get", f"{base}/users", params=params, headers=bearer())
        users = r.get("users") or r.get("items") or r.get("result") or []
        total = r.get("total", None)

        for u in users:
            uname = u.get("username") or "-"
            if args.contains and args.contains not in uname:
                continue
            if args.proto and args.proto not in (list((u.get("proxies") or {}).keys())):
                continue
            rows.append(u)

        printed += len(users)
        if not users: break
        if total is not None and printed >= total: break
        if len(users) < limit: break
        offset += limit
        if offset >= args.max_offset: break

    if args.json:
        print(json.dumps(rows, ensure_ascii=False))
        return

    # === Helper: ringkas kolom Protocol sesuai aturanmu ===
    def summarize_protocol(u):
        # prefer 'inbounds' untuk tahu plugin; fallback ke 'proxies'
        inb = u.get("inbounds") or {}
        proxies = u.get("proxies") or {}
        # hanya protokol yang kita kenal preset-nya
        known_protos = [p for p in INBOUND_PRESETS.keys()]
        present = {p: set(inb.get(p, [])) for p in known_protos if inb.get(p)}

        # kalau inbounds kosong: fallback ke daftar proxies
        if not present:
            keys = list(proxies.keys())
            if not keys:
                return "-"
            if len(keys) == 1:
                return keys[0].upper()
            return " ".join(k.upper() for k in keys)

        # cek "All Protocol" (semua proto dikenal & semua plugin aktif)
        all_proto_ok = True
        for p in known_protos:
            grp = INBOUND_PRESETS.get(p, {})
            allset = set().union(*[set(v) for v in grp.values()]) if grp else set()
            sels = set(inb.get(p, []))
            if not allset or sels != allset:
                all_proto_ok = False
                break
        if all_proto_ok and set(present.keys()) == set(known_protos):
            return "All Protocol"

        # satu proto?
        if len(present) == 1:
            p = next(iter(present))
            sels = present[p]
            grp = INBOUND_PRESETS.get(p, {})
            allset = set().union(*[set(v) for v in grp.values()]) if grp else set()
            if sels == allset and allset:
                return f"{p.upper()} [all]"
            # deteksi single plugin group
            touched = [g for g, lst in grp.items() if sels and sels.issubset(set(lst))]
            if len(touched) == 1:
                return f"{p.upper()} [{touched[0]}]"
            # tampilkan ringkas grup yang tersentuh
            touched_any = [g for g, lst in grp.items() if sels.intersection(lst)]
            if touched_any:
                return f"{p.upper()} [{','.join(touched_any)}]"
            return p.upper()

        # multi proto → ringkas per proto
        parts = []
        for p, sels in present.items():
            grp = INBOUND_PRESETS.get(p, {})
            allset = set().union(*[set(v) for v in grp.values()]) if grp else set()
            if sels == allset and allset:
                parts.append(f"{p.upper()}[all]")
            else:
                touched = [g for g, lst in grp.items() if sels.intersection(lst)]
                parts.append(f"{p.upper()}[{','.join(touched)}]" if touched else p.upper())
        return " ".join(parts)

    # ====== TABEL DENGAN BORDER ======
    header = [
        "No.", "Username", "Protocol", "Status",
        "Penggunaan Data", "Total Penggunaan Data",
        "Data Limit", "Reset Strategy", "Max Dev", "Expires At", "Lock Remaining"
    ]
    #                 No  User  Proto  Status  Used           Life                 Limit    Reset          Max   Expire                    Lock
    widths = [3,       13,   27,   10,     17,             21,                  12,      14,            7,    26,                       14]

    use_ascii = getattr(args, "ascii", False)
    CH = {
        "h": "-" if use_ascii else "─",
        "v": "|" if use_ascii else "│",
        "tl": "+" if use_ascii else "┌",
        "tr": "+" if use_ascii else "┐",
        "bl": "+" if use_ascii else "└",
        "br": "+" if use_ascii else "┘",
        "tm": "+" if use_ascii else "┬",
        "mm": "+" if use_ascii else "┼",
        "bm": "+" if use_ascii else "┴",
    }

    def trunc_ellips(s, w):
        s = str(s)
        if len(s) <= w: return s + " " * (w - len(s))
        if w <= 1: return s[:w]
        return s[: w - 1] + "…"

    def fmt_cell(text, w, align="left"):
        s = str(text)
        if len(s) > w:
            s = trunc_ellips(s, w)
        pad = " " * (w - len(s))
        return (pad + s) if align == "right" else (s + pad)

    def hline(left, mid, right):
        return left + CH["tm"].join(CH["h"] * w for w in widths).replace(CH["tm"], mid) + right

    top    = CH["tl"] + CH["tm"].join(CH["h"] * w for w in widths) + CH["tr"]
    sep    = CH["tl"] + CH["mm"].join(CH["h"] * w for w in widths) + CH["tr"]  # header separator
    middle = CH["tl"] + CH["mm"].join(CH["h"] * w for w in widths) + CH["tr"]  # (tidak dipakai, siap kalau mau)
    bottom = CH["bl"] + CH["bm"].join(CH["h"] * w for w in widths) + CH["br"]

    print(top)
    # header row
    header_cells = []
    for i, h in enumerate(header):
        align = "right" if i == 0 else "left"
        header_cells.append(fmt_cell(h, widths[i], align))
    print(CH["v"] + CH["v"].join(header_cells) + CH["v"])
    print(sep)

    maxdev_map = load_maxdev()

    def lock_remaining(uname):
        p = pathlib.Path(LOCK_DIR) / f"{uname}.json"
        if not p.exists(): return "-"
        try:
            d = json.loads(p.read_text())
            until = int(d.get("locked_until", 0))
            if until <= 0: return "-"
            remain = until - int(time.time())
            if remain <= 0: return "-"
            days = math.floor(remain / 86400)
            if days >= 1: return f"({days}d)"
            hours = math.floor(remain / 3600)
            if hours >= 1: return f"({hours}h)"
            mins = math.floor(remain / 60)
            return f"({mins}m)"
        except Exception:
            return "-"

    for i, u in enumerate(rows, 1):
        uname = u.get("username") or "-"
        status = (u.get("status") or "-").capitalize()
        used = u.get("used_traffic") or 0
        used_life = u.get("lifetime_used_traffic") or 0
        dlim_val = int(u.get("data_limit") or 0)
        limit_str = "Unlimited" if dlim_val == 0 else human_bytes(dlim_val)
        reset_str = map_reset(u.get("data_limit_reset_strategy"))
        max_dev = maxdev_map.get(uname, "-")
        exp = u.get("expire") or 0
        if exp:
            try:
                exp_dt = datetime.fromtimestamp(int(exp), tz=timezone.utc)
                exp_s = exp_dt.strftime("%Y-%m-%d %H:%M:%SZ")
            except Exception:
                exp_s = str(exp)
        else:
            exp_s = "-"
        remain_s = lock_remaining(uname)
        proto_str = summarize_protocol(u)  # ← dari versi sebelumnya

        row = [
            fmt_cell(i, widths[0], "right"),
            fmt_cell(uname, widths[1]),
            fmt_cell(proto_str, widths[2]),
            fmt_cell(status, widths[3]),
            fmt_cell(human_bytes(used), widths[4]),
            fmt_cell(human_bytes(used_life), widths[5]),
            fmt_cell(limit_str, widths[6]),
            fmt_cell(reset_str, widths[7]),
            fmt_cell(max_dev, widths[8]),
            fmt_cell(exp_s, widths[9]),
            fmt_cell(remain_s, widths[10]),
        ]
        print(CH["v"] + CH["v"].join(row) + CH["v"])

    print(bottom)

def sanitize_user_for_put(u):
    allowed = {
        "username", "note",
        "status", "expire",
        "on_hold_timeout", "on_hold_expire_duration",
        "data_limit", "data_limit_reset_strategy",
        "proxies", "inbounds",
        "excluded_inbounds",
    }
    clean = {}
    for k, v in u.items():
        if k in allowed:
            clean[k] = v
    clean.setdefault("username", u.get("username"))
    clean.setdefault("proxies", u.get("proxies") or {})
    clean.setdefault("inbounds", u.get("inbounds") or {})
    clean.setdefault("data_limit", u.get("data_limit") or 0)
    clean.setdefault("data_limit_reset_strategy", u.get("data_limit_reset_strategy") or "no_reset")
    clean.setdefault("status", u.get("status") or "active")
    clean.setdefault("expire", u.get("expire") or 0)
    return clean

def cmd_modify(args):
    ensure_dirs()
    base = api_base()
    current = get_user(base, args.username)
    put_body = sanitize_user_for_put(current)

    # note
    if args.note is not None:
        put_body["note"] = args.note

    # on_hold toggle
    if args.on_hold is not None:
        if args.on_hold:
            put_body["status"] = "on_hold"
            put_body["on_hold_timeout"] = iso_in(args.hold_days)
            put_body["on_hold_expire_duration"] = int((args.days or 0) * 24 * 3600) if args.days else 0
        else:
            put_body["status"] = "active"
            put_body["on_hold_timeout"] = None
            put_body["on_hold_expire_duration"] = 0

    # expire
    if args.always_on:
        put_body["expire"] = 0
    elif args.days is not None:
        put_body["expire"] = expire_ts(args.days) or 0

    # data limit
    if args.unlimited:
        put_body["data_limit"] = 0
    elif args.quota_gb is not None:
        if args.quota_gb <= 0:
            print("GAGAL: --quota-gb harus > 0", file=sys.stderr); sys.exit(1)
        put_body["data_limit"] = int(args.quota_gb * (1024**3))

    if args.reset is not None:
        put_body["data_limit_reset_strategy"] = args.reset

    # ensure containers
    put_body.setdefault("proxies", {})
    put_body.setdefault("inbounds", {})

    # proto switch
    if args.proto:
        target = args.proto
        # proxies
        if target == "vmess":
            if args.regen_id or "vmess" not in put_body["proxies"]:
                put_body["proxies"]["vmess"] = {"id": str(uuidlib.uuid4())}
        elif target == "vless":
            v = put_body["proxies"].get("vless", {})
            if args.regen_id or "id" not in v:
                v["id"] = str(uuidlib.uuid4())
            if args.set_flow:
                v["flow"] = args.set_flow
            put_body["proxies"]["vless"] = v
        elif target == "trojan":
            if args.regen_pass or "trojan" not in put_body["proxies"]:
                put_body["proxies"]["trojan"] = {"password": str(uuidlib.uuid4())}

        # inbounds
        if args.plugin:
            put_body["inbounds"][target] = build_inbounds(target, args.plugin)[target]
        else:
            put_body["inbounds"][target] = build_inbounds(target, ["all"])[target]

        # drop others unless kept
        if not getattr(args, "keep_other_protos", False):
            for k in list(put_body["proxies"].keys()):
                if k != target: del put_body["proxies"][k]
            for k in list(put_body["inbounds"].keys()):
                if k != target: del put_body["inbounds"][k]

    # set_flow tanpa ganti proto
    if args.set_flow and (args.proto == "vless" or ("vless" in put_body["proxies"])):
        v = put_body["proxies"].get("vless", {})
        v["flow"] = args.set_flow
        put_body["proxies"]["vless"] = v

    # max-dev metadata
    if args.max_dev is not None:
        set_user_maxdev(args.username, args.max_dev)

    # PUT
    resp = http_json("put", f"{base}/user/{args.username}", headers=bearer(), data=json.dumps(put_body))
    resp_protos = list((resp.get("proxies") or {}).keys())
    first_proto = args.proto or (resp_protos[0] if resp_protos else "vmess")
    save_response(args.username, first_proto, resp, args.out_dir)

    if args.json:
        print(json.dumps(resp, ensure_ascii=False))
    else:
        if args.quiet:
            print("OK")
        else:
            md = get_user_maxdev(args.username, "-")
            print(f"OK modified username={args.username} max-dev={md}")

def cmd_purge_all(args):
    base = api_base()
    warning = "PERINGATAN: Ini akan MENGHAPUS SEMUA USER dari Marzban!"
    confirm_or_exit(warning, args.force)

    total_deleted = 0
    offset = 0
    while True:
        r = http_json("get", f"{base}/users", params={"limit": 200, "offset": offset}, headers=bearer())
        users = r.get("users") or []
        if not users:
            break
        for u in users:
            uname = u.get("username")
            if not uname:
                continue
            if args.dry_run:
                print(f"DRY-RUN delete {uname}")
                continue
            rr = http_raw("delete", f"{base}/user/{uname}", headers=bearer())
            if rr.ok:
                total_deleted += 1
                set_user_maxdev(uname, None)
                remove_snapshot(uname)
                if not args.quiet:
                    print(f"deleted {uname}")
        if len(users) < 200:
            break
        offset += 200
    if args.dry_run:
        print("DRY-RUN selesai.")
    else:
        print(f"OK purge-all: {total_deleted} user terhapus.")

def cmd_purge_expired(args):
    base = api_base()
    warning = "PERINGATAN: Ini akan menghapus semua user yang SUDAH EXPIRED."
    confirm_or_exit(warning, args.force)

    now_ts = int(time.time())
    total_deleted = 0
    offset = 0
    while True:
        r = http_json("get", f"{base}/users", params={"limit": 200, "offset": offset}, headers=bearer())
        users = r.get("users") or []
        if not users:
            break
        for u in users:
            uname = u.get("username"); exp = u.get("expire") or 0
            if not uname or not exp or int(exp) == 0:
                continue
            if int(exp) > 0 and int(exp) <= now_ts:
                if args.dry_run:
                    print(f"DRY-RUN delete expired {uname}")
                    continue
                rr = http_raw("delete", f"{base}/user/{uname}", headers=bearer())
                if rr.ok:
                    total_deleted += 1
                    set_user_maxdev(uname, None)
                    remove_snapshot(uname)
                    if not args.quiet:
                        print(f"deleted {uname}")
        if len(users) < 200:
            break
        offset += 200
    if args.dry_run:
        print("DRY-RUN selesai.")
    else:
        print(f"OK purge-expired: {total_deleted} user terhapus.")

def cmd_ips(args):
    ensure_dirs()

    # 1) Ambil user dari API sesuai filter
    proto = args.proto
    users = fetch_users_filtered(proto=proto, contains=args.contains, only_active=args.only_active)
    usernames = [u.get("username") for u in users if u.get("username")]
    if not usernames:
        print("Tidak ada user yang cocok filter.")
        return

    # 2) Index by username: simpan proto(s), online_at, hitungan log dan set IP
    info = {}
    for u in users:
        uname = u["username"]
        info[uname] = {
            "protos": list((u.get("proxies") or {}).keys()),
            "online_at": to_wib_str(u.get("online_at")),
            "ips": set(),
            "log_count": 0,
        }

    # 3) Scan access.log sekali, mapping username→IP unik
    log_file = "/var/lib/marzban/assets/access.log"
    name_re = build_username_regex(usernames)
    if not os.path.exists(log_file):
        print(f"log tidak ditemukan: {log_file}", file=sys.stderr)
        return

    with open(log_file, "r", errors="ignore") as f:
        for line in f:
            m_ip = IP_RE.search(line)
            if not m_ip:
                continue
            ip = m_ip.group(1)
            if is_private_ip(ip):
                continue
            if name_re:
                matches = name_re.findall(line)
                if not matches:
                    continue
                for uname in set(matches):
                    if uname in info:
                        info[uname]["ips"].add(ip)
                        info[uname]["log_count"] += 1

    # 4) Tampilkan
    total_subnets = 0  # Hitung berdasarkan subnet /16 (2 oktet pertama)

    for i, uname in enumerate(sorted(info.keys()), 1):
        rec = info[uname]
        protos = ",".join(rec["protos"]) if rec["protos"] else "-"
        print(f"{i}. User: {uname} (Protocols: {protos})")

        if rec["online_at"] == "null":
            print("   No data available for this user.")
        else:
            print(f"   Data login terakhir (WIB): \033[94m{rec['online_at']}\033[0m")

            if not rec["ips"]:
                print("   No connected IPs for this user.")
            else:
                # Group IPs berdasarkan 2 oktet pertama (subnet /16)
                subnet_groups = {}
                for ip in rec["ips"]:
                    parts = ip.split(".")
                    if len(parts) == 4:
                        # Ambil 2 oktet pertama sebagai key
                        subnet_key = f"{parts[0]}.{parts[1]}"
                        if subnet_key not in subnet_groups:
                            subnet_groups[subnet_key] = []
                        subnet_groups[subnet_key].append(ip)

                # Tampilkan hanya 1 IP pertama dari setiap subnet
                print("   Connected IPs:")
                for idx, (subnet_key, ips) in enumerate(sorted(subnet_groups.items()), 1):
                    # Ambil IP pertama dari grup (atau bisa random)
                    representative_ip = sorted(ips)[0]

                    if args.asn:
                        org = ipinfo_org(representative_ip) or "N/A"
                        print(f"      {idx}. {representative_ip}")
                        print(f"         ASN/ISP: {org}")
                    else:
                        print(f"      {idx}. {representative_ip}")

                # Hitung total subnet unik (bukan total IP)
                total_subnets += len(subnet_groups)

            # log count ringkas
            if rec["log_count"] == 0:
                print("   Log data pada user ini: No log data available for this user.")
            else:
                print(f"   Log data pada user ini: User appears in log \033[0;33m{rec['log_count']} times.\033[0m")

        print("------------------------")

    # Ringkasan - hitung berdasarkan subnet, bukan IP individual
    if proto == "all":
        print(f"Total Connected Subnets (ALL protos): {total_subnets}")
    else:
        print(f"Total Connected Subnets for {proto}: {total_subnets}")

    # 5) JSON (opsional)
    if args.json:
        out = []
        for uname, rec in info.items():
            ips_sorted = sorted(list(rec["ips"]))

            # Group by subnet /16
            subnet_groups = {}
            for ip in ips_sorted:
                parts = ip.split(".")
                if len(parts) == 4:
                    subnet_key = f"{parts[0]}.{parts[1]}"
                    if subnet_key not in subnet_groups:
                        subnet_groups[subnet_key] = []
                    subnet_groups[subnet_key].append(ip)

            # Ambil 1 IP per subnet
            representative_ips = [sorted(ips)[0] for ips in subnet_groups.values()]

            out.append({
                "username": uname,
                "protos": rec["protos"],
                "online_at_wib": rec["online_at"],
                "all_ips": ips_sorted,
                "representative_ips": sorted(representative_ips),
                "unique_subnets": len(subnet_groups),
                "log_count": rec["log_count"],
            })
        print(json.dumps(out, ensure_ascii=False))

def cmd_status(args):
    ensure_dirs()
    base = api_base()

    # ANSI colors (mirip bash-mu)
    RED = '\033[0;31m'
    GREEN = '\033[0;32m'
    YELLOW = '\033[0;33m'
    CYAN = '\033[0;36m'
    NC = '\033[0m'

    # TZ WIB
    try:
        from datetime import timezone, timedelta, datetime
    except Exception:
        pass
    ASIA_JAKARTA = timezone(timedelta(hours=7))

    def to_wib(iso_str):
        if not iso_str or iso_str == "null":
            return "-"
        try:
            dt = datetime.fromisoformat(iso_str.replace("Z", "+00:00"))
            return dt.astimezone(ASIA_JAKARTA).strftime("%Y-%m-%d %H:%M:%S")
        except Exception:
            return iso_str

    def expire_wib(exp_val):
        if not exp_val:
            return "Always ON"
        try:
            ts = int(exp_val)
            if ts <= 0:
                return "Always ON"
            dt = datetime.fromtimestamp(ts, tz=ASIA_JAKARTA)
            return dt.strftime("%Y-%m-%d %H:%M:%S")
        except Exception:
            return str(exp_val)

    # GET user
    try:
        u = http_json("get", f"{base}/user/{args.username}", headers=bearer())
    except SystemExit:
        raise
    except Exception as e:
        print(f"Error: gagal ambil user '{args.username}': {e}", file=sys.stderr)
        sys.exit(1)

    # handle user not found
    if isinstance(u, dict) and u.get("detail") == "User not found":
        print(f"Error: Pengguna '{args.username}' tidak ditemukan.", file=sys.stderr)
        sys.exit(1)

    # ambil field
    uname = u.get("username") or args.username
    status_raw = (u.get("status") or "-").lower()
    created_at = to_wib(u.get("created_at"))
    reset_strategy = (u.get("data_limit_reset_strategy") or "no_reset")

    note_val = u.get("note")
    note_human = "Tidak ada catatan disini" if (note_val is None or note_val == "null") else str(note_val)

    used = u.get("used_traffic") or 0
    used_life = u.get("lifetime_used_traffic") or 0
    dlim = u.get("data_limit")
    dlim_h = "Unlimited" if (dlim is None or int(dlim) == 0) else human_bytes(dlim)

    # protocol: ambil key pertama dari proxies (seperti skrip bash)
    proxies = u.get("proxies") or {}
    keys = list(proxies.keys())
    proto = (keys[0] if keys else "-").replace("shadowsocks", "ss")

    # map reset
    reset_map = {
        "no_reset": "loss_doll",
        "day": "harian",
        "week": "mingguan",
        "month": "bulanan",
        "year": "tahunan",
    }
    reset_label = reset_map.get(reset_strategy, reset_strategy)

    # map status + warna
    if status_raw == "active":
        status_col = f"{GREEN}Active{NC}"
    elif status_raw == "disabled":
        status_col = f"{CYAN}Disabled{NC}"
    elif status_raw == "on_hold":
        status_col = f"{CYAN}On_Hold{NC}"
    elif status_raw in ("limited", "expired"):
        status_col = f"{RED}{status_raw.capitalize()}{NC}"
    else:
        status_col = status_raw.capitalize()

    # expire → WIB
    exp_h = expire_wib(u.get("expire"))

    # human-bytes
    used_h = human_bytes(used)
    life_h = human_bytes(used_life)

    if args.json:
        out = {
            "username": uname,
            "protocol": proto,
            "status": status_raw,
            "used_traffic": used,
            "used_traffic_human": used_h,
            "lifetime_used_traffic": used_life,
            "lifetime_used_traffic_human": life_h,
            "data_limit": dlim,
            "data_limit_human": dlim_h,
            "reset_strategy": reset_strategy,
            "reset_strategy_label": reset_label,
            "created_at_wib": created_at,
            "note": note_human,
            "expire_wib": exp_h,
        }
        print(json.dumps(out, ensure_ascii=False))
        return

    # cetak seperti banner bash
    print("╭───────────────────────────╮")
    print(f"│ Username: {uname}")
    print(f"│ Protocol Xray: {proto}")
    print(f"│ Status: {status_col}")
    print(f"│ Data Limit: {used_h} / {dlim_h}")
    print(f"│ Total Penggunaan Data: {life_h}")
    print(f"│ Reset strategy: {reset_label}")
    print(f"│ Akun dibuat: {created_at}")
    print(f"│ Note: {note_human}")
    print(f"│ Expired: {exp_h}")
    print("╰───────────────────────────╯")

def cmd_extend(args):
    ensure_dirs()
    base = api_base()

    if args.days is None and args.hours is None:
        print("GAGAL: butuh --days atau --hours", file=sys.stderr)
        sys.exit(1)
    add_sec = int(args.days or 0) * 86400 + int(args.hours or 0) * 3600
    if add_sec <= 0:
        print("GAGAL: durasi perpanjangan harus > 0", file=sys.stderr)
        sys.exit(1)

    # get user & siapkan body PUT
    cur = get_user(base, args.username)
    put_body = sanitize_user_for_put(cur)

    now_ts = int(time.time())
    cur_exp = int(cur.get("expire") or 0)
    base_ts = now_ts if (cur_exp == 0 or cur_exp <= now_ts) else cur_exp
    new_exp = base_ts + add_sec
    put_body["expire"] = new_exp

    resp = http_json("put", f"{base}/user/{args.username}", headers=bearer(), data=json.dumps(put_body))

    if args.json:
        out = {
            "username": args.username,
            "old_expire": cur_exp,
            "new_expire": new_exp,
            "old_expire_utc": datetime.fromtimestamp(cur_exp, tz=timezone.utc).strftime("%Y-%m-%d %H:%M:%SZ") if cur_exp else "AlwaysON/Expired",
            "new_expire_utc": datetime.fromtimestamp(new_exp, tz=timezone.utc).strftime("%Y-%m-%d %H:%M:%SZ"),
            "added_seconds": add_sec
        }
        print(json.dumps(out, ensure_ascii=False))
    else:
        if args.quiet:
            print("OK")
        else:
            old_s = datetime.fromtimestamp(cur_exp, tz=timezone.utc).strftime("%Y-%m-%d %H:%M:%SZ") if cur_exp else "AlwaysON/Expired"
            new_s = datetime.fromtimestamp(new_exp, tz=timezone.utc).strftime("%Y-%m-%d %H:%M:%SZ")
            print(f"OK extend {args.username}: {old_s}  →  {new_s}  (+{add_sec//86400}d { (add_sec%86400)//3600 }h)")

def cmd_reset_quota(args):
    ensure_dirs()
    base = api_base()
    uname = args.username

    url = f"{base}/user/{uname}/reset"
    try:
        r = requests.post(url, headers=bearer(), timeout=20)
        if not r.ok:
            # beberapa server balas 204; kalau bukan ok, anggap gagal
            print(f"GAGAL reset usage {uname}: {r.status_code} {r.text[:160]}", file=sys.stderr)
            sys.exit(1)
    except Exception as e:
        print(f"GAGAL reset usage {uname}: {e}", file=sys.stderr)
        sys.exit(1)

    if args.json:
        print(json.dumps({"username": uname, "reset_usage": "OK"}, ensure_ascii=False))
    else:
        print("OK" if args.quiet else f"OK reset usage {uname}")

def cmd_maxdev_sync(args):
    ensure_dirs()
    base = api_base()

    # ambil semua user (pagination)
    limit = 200
    offset = 0
    users = []
    while True:
        params = {"limit": limit, "offset": offset}
        if args.status:
            params["status"] = args.status
        r = http_json("get", f"{base}/users", params=params, headers=bearer())
        batch = r.get("users") or []
        if not batch:
            break
        users.extend(batch)
        if len(batch) < limit:
            break
        offset += limit

    # filter lanjutan
    def match_proto(u):
        if args.proto in (None, "all"):
            return True
        protos = list((u.get("proxies") or {}).keys())
        return args.proto in protos

    def match_contains(u):
        if not args.contains:
            return True
        uname = (u.get("username") or "")
        return args.contains in uname

    users = [u for u in users if match_proto(u) and match_contains(u)]

    if not users:
        if not args.quiet:
            print("Tidak ada user yang cocok filter.")
        return

    # muat/ubah metadata
    current = load_maxdev()
    changed = 0
    skipped_have = 0

    for u in users:
        uname = u.get("username")
        if not uname:
            continue
        if args.only_missing and uname in current:
            skipped_have += 1
            continue
        # set nilai
        if not args.dry_run:
            current[uname] = int(args.value)
        changed += 1

        if not args.quiet:
            print(f"{'[DRY] ' if args.dry_run else ''}set max-dev {uname} = {args.value}")

    # simpan
    if not args.dry_run:
        save_maxdev(current)

    if not args.quiet:
        print(f"OK maxdev-sync: target={len(users)} changed={changed} skipped_existing={skipped_have} dry_run={args.dry_run}")

# ---------- cli ----------
def build_parser():
    p = argparse.ArgumentParser(prog="lingvpn_mz", description="LingVPN Marzban Manager (add/delete/list/modify + per-user max-dev)")
    sub = p.add_subparsers(dest="cmd", required=True)

    # add
    a = sub.add_parser("add", help="Tambah user (vmess/vless/trojan)")
    a.add_argument("username")
    a.add_argument("--proto", choices=["vmess","vless","trojan"], default="vmess")
    a.add_argument("--plugin", choices=["ws","grpc","hu","reality","all"], nargs="+", default=["all"],
                   help="Pilih plugin (bisa multi). 'reality' (VLESS_REALITY_* + flow vision)")
    gq = a.add_mutually_exclusive_group(required=True)
    gq.add_argument("--quota-gb", type=int, help="Kuota GB (>0)")
    gq.add_argument("--unlimited", action="store_true", help="Tanpa batas kuota (data_limit=0)")
    a.add_argument("--reset", choices=["no_reset","day","week","month","year"], default="no_reset")
    gd = a.add_mutually_exclusive_group(required=True)
    gd.add_argument("--days", type=int, help="Masa aktif (hari). Atau pakai --always-on")
    gd.add_argument("--always-on", dest="days", action="store_const", const=None, help="Tanpa expired")
    a.add_argument("--on-hold", action="store_true", help="Status on_hold + jendela aktivasi")
    a.add_argument("--hold-days", type=int, default=14, help="Durasi on_hold_timeout (hari)")
    a.add_argument("--note", default="", help="Catatan user")
    a.add_argument("--max-dev", type=int, help="Batas device per user (metadata lokal)")
    a.add_argument("--out-dir", default=DATA_DIR, help="Folder penyimpanan (default: /var/lib/marzban)")
    a.add_argument("--quiet", action="store_true", help="Output hanya 'OK' atau 'GAGAL'")
    a.add_argument("--json", action="store_true", help="Print JSON respons (panjang)")
    a.set_defaults(func=cmd_add)

    # delete
    d = sub.add_parser("delete", help="Hapus user berdasarkan username")
    d.add_argument("username")
    d.add_argument("--out-dir", default=DATA_DIR, help="Folder penyimpanan (untuk hapus snapshot)")
    d.add_argument("--quiet", action="store_true")
    d.set_defaults(func=cmd_delete)

    # list
    l = sub.add_parser("list", help="Daftar user (tabel ringkas, termasuk Max Dev & Lock Remaining)")
    l.add_argument("--limit", type=int, default=100)
    l.add_argument("--max-offset", type=int, default=5000)
    l.add_argument("--status", choices=["active","disabled","on_hold"])
    l.add_argument("--proto", choices=["vmess","vless","trojan"])
    l.add_argument("--username")
    l.add_argument("--contains")
    l.add_argument("--json", action="store_true")
    l.set_defaults(func=cmd_list)

    # modify (PUT)
    m = sub.add_parser("modify", help="Ubah properti user (PUT)")
    m.add_argument("username")
    m.add_argument("--note")
    m.add_argument("--on-hold", dest="on_hold", action="store_true")
    m.add_argument("--no-on-hold", dest="on_hold", action="store_false")
    m.add_argument("--hold-days", type=int, default=14)
    m.add_argument("--days", type=int)
    m.add_argument("--always-on", action="store_true")
    gqm = m.add_mutually_exclusive_group()
    gqm.add_argument("--quota-gb", type=int)
    gqm.add_argument("--unlimited", action="store_true")
    m.add_argument("--reset", choices=["no_reset","day","week","month","year"])
    m.add_argument("--proto", choices=["vmess","vless","trojan"])
    m.add_argument("--plugin", choices=["ws","grpc","hu","reality","all"], nargs="+")
    m.add_argument("--regen-id", action="store_true")
    m.add_argument("--regen-pass", action="store_true")
    m.add_argument("--set-flow", choices=["xtls-rprx-vision"])
    m.add_argument("--max-dev", type=int, help="Set/ubah batas device per user (metadata)")
    m.add_argument("--out-dir", default=DATA_DIR)
    m.add_argument("--quiet", action="store_true")
    m.add_argument("--json", action="store_true")
    m.set_defaults(func=cmd_modify)
    m.add_argument("--keep-other-protos", action="store_true",
               help="Jangan hapus proto lain saat --proto diberikan (default: hapus)")

    # enable
    e = sub.add_parser("enable", help="Aktifkan user (status=active)")
    e.add_argument("username")
    e.set_defaults(func=cmd_enable)

    # disable
    x = sub.add_parser("disable", help="Nonaktifkan user (status=disabled)")
    x.add_argument("username")
    x.set_defaults(func=cmd_disable)

    # purge-all
    pa = sub.add_parser("purge-all", help="Hapus SEMUA user (butuh konfirmasi)")
    pa.add_argument("--force", action="store_true", help="Lewati konfirmasi (bahaya)")
    pa.add_argument("--dry-run", action="store_true", help="Simulasi saja, tidak menghapus")
    pa.add_argument("--quiet", action="store_true")
    pa.set_defaults(func=cmd_purge_all)

    # purge-expired
    pe = sub.add_parser("purge-expired", help="Hapus semua user yang sudah expired (butuh konfirmasi)")
    pe.add_argument("--force", action="store_true")
    pe.add_argument("--dry-run", action="store_true")
    pe.add_argument("--quiet", action="store_true")
    pe.set_defaults(func=cmd_purge_expired)

    # ips
    ipsp = sub.add_parser("ips", help="Cek IP terhubung per user dari access.log (optional ASN)")
    ipsp.add_argument("--proto", choices=["vmess","vless","trojan","shadowsocks","all"], default="all")
    ipsp.add_argument("--contains", help="Filter username yang mengandung teks")
    ipsp.add_argument("--only-active", action="store_true", help="Hanya user status active")
    ipsp.add_argument("--asn", action="store_true", help="Lookup ASN/ISP via ipinfo.io")
    ipsp.add_argument("--json", action="store_true", help="Output JSON")
    ipsp.set_defaults(func=cmd_ips)

    # status
    st = sub.add_parser("status", help="Tampilkan status detail 1 user (WIB, warna, human-bytes)")
    st.add_argument("username")
    st.add_argument("--json", action="store_true", help="Output JSON")
    st.set_defaults(func=cmd_status)

    # extend (tambah masa aktif dari expire/now)
    ex = sub.add_parser("extend", help="Tambah masa aktif (bukan replace)")
    ex.add_argument("username")
    ex.add_argument("--days", type=int, help="Jumlah hari yang ditambahkan")
    ex.add_argument("--hours", type=int, help="Jumlah jam yang ditambahkan")
    ex.add_argument("--quiet", action="store_true")
    ex.add_argument("--json", action="store_true")
    ex.set_defaults(func=cmd_extend)

    # reset-quota (reset used_traffic ke 0)
    rq = sub.add_parser("reset-quota", help="Reset pemakaian (used_traffic) pengguna")
    rq.add_argument("username")
    rq.add_argument("--quiet", action="store_true")
    rq.add_argument("--json", action="store_true")
    rq.set_defaults(func=cmd_reset_quota)

    # maxdev-sync
    mds = sub.add_parser("maxdev-sync", help="Set Max Devices (metadata lokal) massal")
    mds.add_argument("--value", type=int, required=True, help="Nilai Max Devices yang akan diterapkan")
    mds.add_argument("--only-missing", action="store_true",
                     help="Hanya mengisi user yang belum punya Max Devices (default: False)")
    mds.add_argument("--proto", choices=["vmess","vless","trojan","shadowsocks","all"], default="all",
                     help="Filter berdasarkan proto (default: all)")
    mds.add_argument("--status", choices=["active","disabled","on_hold"],
                     help="Filter berdasarkan status user")
    mds.add_argument("--contains", help="Filter username yang mengandung teks")
    mds.add_argument("--dry-run", action="store_true", help="Simulasi saja, tidak menyimpan perubahan")
    mds.add_argument("--quiet", action="store_true", help="Minimalkan output")
    mds.set_defaults(func=cmd_maxdev_sync)

    return p

def main():
    parser = build_parser()
    args = parser.parse_args()
    try:
        args.func(args)
    except KeyboardInterrupt:
        print("GAGAL: dibatalkan", file=sys.stderr)
        sys.exit(130)

if __name__ == "__main__":
    main()
