#!/usr/bin/env python3
# lvbackup-menu.py — LingVPN + Marzban (Docker/Nginx) Backup & Restore via Telegram
# UI panel mirip menu LingVPN lainnya.
# Catatan:
# - ZIP ber-password tetap via util "zip/unzip" (subprocess) supaya kompatibel.
# - Kirim Telegram pakai requests; bisa diganti curl jika mau.

import os, sys, shutil, subprocess, json, time, re
from datetime import datetime
from pathlib import Path
try:
    import requests
except Exception:
    print("Butuh paket: python3-requests")
    sys.exit(1)

# ================== Konstanta/Path ==================
CFG_TG      = Path("/root/telegram_config.conf")
PASS_FILE   = Path("/root/passbackup")
FILEID_LOG  = Path("/root/file_id.txt")
LOG_BACKUP  = Path("/root/log-backup.txt")
NAME_FILE   = Path("/root/nama")

DATE_YMD    = datetime.now().strftime("%Y%m%d")
DATE_HMS    = datetime.now().strftime("%H%M%S")
DATE_DASH   = datetime.now().strftime("%m-%d-%Y")

# ================== UI Panel (mirip TUI lain) ==================
import shutil as _shutil
def _hr(w): return "─" * max(0, w)
def _center(w, s):
    s = s or ""
    ln = len(s)
    if ln >= w: return s
    left = (w - ln) // 2
    return " " * left + s + " " * (w - ln - left)

def panel(title, body_lines):
    os.system("clear")
    cols = _shutil.get_terminal_size(fallback=(100,28)).columns
    w = max(60, min(90, cols-4))
    top = "┌" + _hr(w) + "┐"
    mid = "├" + _hr(w) + "┤"
    bot = "└" + _hr(w) + "┘"
    print(top)
    print("│ " + _center(w-2, title) + " │")
    print(mid)
    for ln in body_lines:
        # potong panjang
        if len(ln) > w-2:
            ln = ln[:w-5] + "..."
        print("│ " + ln.ljust(w-2) + " │")
    print(bot)

def pause():
    try:
        input("\n[Enter] untuk kembali...")
    except KeyboardInterrupt:
        pass

def ask(prompt):
    try:
        return input(prompt)
    except KeyboardInterrupt:
        return ""

# ================== Utils ==================
def need(cmd):
    if shutil.which(cmd) is None:
        panel("Dependensi Hilang", [f"butuh command: {cmd}", "Install dan coba lagi."])
        sys.exit(1)

for c in ("zip","unzip","jq","crontab","docker"):
    # jq & docker optional; kalau tak ada, fungsi yang pakai akan fallback/di-skip
    if c in ("zip","unzip","crontab"):
        need(c)

def ensure_tg_config():
    if not CFG_TG.exists():
        panel("Telegram Config", [
            "Bot token & chat ID belum ada.",
            "Masukkan kredensial bot Telegram untuk disimpan.",
        ])
        bot = ask("Telegram bot token: ").strip()
        chat = ask("Telegram chat ID : ").strip()
        CFG_TG.write_text(f"botToken={bot}\nchatId={chat}\n")
        panel("Simpan", [f"Disimpan: {CFG_TG}"])
        sys.exit(0)
    # parse
    bot = chat = None
    for ln in CFG_TG.read_text().splitlines():
        ln = ln.strip()
        if ln.startswith("botToken="): bot = ln.split("=",1)[1].strip()
        if ln.startswith("chatId="):   chat = ln.split("=",1)[1].strip()
    if not bot or not chat:
        panel("Telegram Config", ["botToken/chatId kosong di file config."])
        sys.exit(1)
    return bot, chat

def ensure_password():
    if not PASS_FILE.exists() or not PASS_FILE.read_text().strip():
        panel("Password ZIP", [f"Buat password ZIP di {PASS_FILE}"])
        sys.exit(1)
    return PASS_FILE.read_text().strip()

def get_name_base():
    return NAME_FILE.read_text().strip() if NAME_FILE.exists() and NAME_FILE.read_text().strip() else "lingvpn"

def run(cmd, check=True, capture=False):
    if capture:
        return subprocess.run(cmd, text=True, capture_output=True, check=check)
    return subprocess.run(cmd, check=check)

def safe_copy(src, dst):
    srcp = Path(src)
    dstp = Path(dst)
    if not srcp.exists(): return
    if srcp.is_dir():
        if dstp.exists():
            shutil.rmtree(dstp)
        shutil.copytree(srcp, dstp)
    else:
        dstp.parent.mkdir(parents=True, exist_ok=True)
        shutil.copy2(srcp, dstp)

# ================== BACKUP ==================
def backup_build_tree() -> Path:
    BKROOT = Path("/root/backup")
    if BKROOT.exists(): shutil.rmtree(BKROOT)
    (BKROOT/"backup").mkdir(parents=True, exist_ok=True)
    B = BKROOT/"backup"

    # Stack Docker marzban (nginx + compose)
    safe_copy("/opt/marzban", B/"opt/marzban")

    # Data marzban
    safe_copy("/var/lib/marzban/db.sqlite3", B/"var/lib/marzban/db.sqlite3")
    safe_copy("/var/lib/marzban/xray_config.json", B/"var/lib/marzban/xray_config.json")

    # Web root + lvsub page
    safe_copy("/var/www/html", B/"var/www/html")
    if Path("/var/www/html/lvsub/index.html").exists():
        (B/"var_www_lvsub").mkdir(parents=True, exist_ok=True)
        safe_copy("/var/www/html/lvsub/index.html", B/"var_www_lvsub/index.html")

    # LingVPN (SSH-WS)
    safe_copy("/etc/lingvpn", B/"etc/lingvpn")
    for f in ("/usr/local/sbin/wsproxy.py",
              "/usr/local/sbin/lvctl",
              "/usr/local/sbin/lvmenu",
              "/usr/local/sbin/lvbackup-menu",
              "/usr/local/sbin/lvsubd.py"):
        safe_copy(f, B/f.split("/usr/local/sbin/")[1])

    safe_copy("/etc/systemd/system/lvsubd.service", B/"etc/systemd/system/lvsubd.service")

    # VNStat + Telegram meta
    safe_copy("/var/lib/vnstat/vnstat.db", B/"var/lib/vnstat/vnstat.db")
    if CFG_TG.exists(): safe_copy(str(CFG_TG), B/"telegram_config.conf")
    if FILEID_LOG.exists(): safe_copy(str(FILEID_LOG), B/"file_id.txt")

    return BKROOT

def backup_and_send_telegram():
    bot, chat = ensure_tg_config()
    pw = ensure_password()
    NAME = get_name_base()
    BASE = f"{NAME}_{DATE_HMS}_{DATE_YMD}"
    ts = datetime.now().strftime("%T")

    panel("Backup", ["Mempersiapkan struktur backup…"])
    BKROOT = backup_build_tree()

    panel("Backup", ["Membuat ZIP (password)…"])
    zipfile = Path(f"/root/{BASE}.zip")
    # zip -rP <pass> zipfile .
    try:
        run(["bash","-lc", f"cd {BKROOT/'backup'} && zip -rP {shlex_quote(pw)} {shlex_quote(str(zipfile))} . >/dev/null"])
    except subprocess.CalledProcessError:
        panel("Error", ["Gagal membuat ZIP. Pastikan paket 'zip' terinstall."])
        return
    # kirim Telegram
    panel("Backup", ["Mengirim ke Telegram…"])
    try:
        with open(zipfile, "rb") as f:
            resp = requests.post(f"https://api.telegram.org/bot{bot}/sendDocument",
                                 data={"chat_id": chat}, files={"document": f}, timeout=60)
        ok = resp.ok
    except Exception as e:
        ok = False

    if ok:
        try:
            file_id = resp.json().get("result",{}).get("document",{}).get("file_id")
        except Exception:
            file_id = None
        if file_id:
            with FILEID_LOG.open("a", encoding="utf-8") as g:
                g.write(f"### {NAME}_{DATE_DASH}\n")
                g.write(f"File ID: {file_id}\n")
            with LOG_BACKUP.open("a", encoding="utf-8") as g:
                g.write(f"Server backup pada {DATE_DASH} {ts}\n")
            panel("Sukses", [f"Terkirim. file_id: {file_id}"])
        else:
            panel("Perhatian", ["Terkirim, tapi tidak dapat membaca file_id dari respons."])
    else:
        panel("Gagal", ["Gagal kirim ke Telegram."])

    shutil.rmtree(BKROOT, ignore_errors=True)
    try: zipfile.unlink()
    except Exception: pass
    pause()

# ================== RESTORE ==================
def chown_execbits():
    for f in ("/usr/local/sbin/wsproxy.py",
              "/usr/local/sbin/lvctl",
              "/usr/local/sbin/lvmenu",
              "/usr/local/sbin/lvbackup-menu",
              "/usr/local/sbin/lvsubd.py"):
        if Path(f).exists():
            try: os.chmod(f, 0o755)
            except Exception: pass

def systemctl(*args):
    try:
        run(["systemctl", *args], check=False)
    except Exception:
        pass

def restore_from_zip(zip_path: str):
    zip_p = Path(zip_path)
    if not zip_p.exists() or zip_p.stat().st_size == 0:
        panel("Restore", [f"ZIP tidak ditemukan: {zip_path}"])
        return
    pw = ensure_password()
    TMP = Path("/root/restore-work")
    shutil.rmtree(TMP, ignore_errors=True)
    TMP.mkdir(parents=True, exist_ok=True)

    panel("Restore", ["Unzip arsip…"])
    # unzip -P <pass> zip -d TMP
    try:
        run(["bash","-lc", f"unzip -P {shlex_quote(pw)} {shlex_quote(str(zip_p))} -d {shlex_quote(str(TMP))} >/dev/null"])
    except subprocess.CalledProcessError:
        panel("Error", ["Unzip gagal (password benar?)."])
        shutil.rmtree(TMP, ignore_errors=True)
        return

    # stop layanan
    systemctl("stop","lvsubd")
    systemctl("stop","wsproxy")

    # docker compose down
    if Path("/opt/marzban/docker-compose.yml").exists():
        panel("Restore", ["Docker compose down (marzban)…"])
        run(["bash","-lc","cd /opt/marzban && docker compose down"], check=False)

    # copy back
    # stack docker
    safe_copy(TMP/"opt/marzban", "/opt/marzban")
    # data
    safe_copy(TMP/"var/lib/marzban/db.sqlite3", "/var/lib/marzban/db.sqlite3")
    safe_copy(TMP/"var/lib/marzban/xray_config.json", "/var/lib/marzban/xray_config.json")
    # web
    safe_copy(TMP/"var/www/html", "/var/www/html")
    if (TMP/"var_www_lvsub/index.html").exists():
        Path("/var/www/html/lvsub").mkdir(parents=True, exist_ok=True)
        safe_copy(TMP/"var_www_lvsub/index.html", "/var/www/html/lvsub/index.html")
    # lingvpn files
    safe_copy(TMP/"etc/lingvpn", "/etc/lingvpn")
    for f in ("wsproxy.py","lvctl","lvmenu","lvbackup-menu","lvsubd.py"):
        src = TMP/f
        if src.exists(): safe_copy(src, f"/usr/local/sbin/{f}")
    safe_copy(TMP/"etc/systemd/system/lvsubd.service", "/etc/systemd/system/lvsubd.service")
    # vnstat + telegram meta
    safe_copy(TMP/"var/lib/vnstat/vnstat.db", "/var/lib/vnstat/vnstat.db")
    if (TMP/"telegram_config.conf").exists():
        safe_copy(TMP/"telegram_config.conf", str(CFG_TG))
    if (TMP/"file_id.txt").exists():
        safe_copy(TMP/"file_id.txt", str(FILEID_LOG))

    # exec bits
    chown_execbits()

    # reload + start
    systemctl("daemon-reload")
    systemctl("restart","lvsubd")
    systemctl("restart","wsproxy")

    # Re-provision SSH-WS dari /etc/lingvpn/db.json (jika ada)
    dbj = Path("/etc/lingvpn/db.json")
    if dbj.exists() and dbj.stat().st_size>0:
        panel("Restore", ["Re-provision SSH-WS users dari /etc/lingvpn/db.json…"])
        try:
            data = json.loads(dbj.read_text())
            for u, inf in (data.get("users") or {}).items():
                pw = str(inf.get("password") or "")
                if pw and pw != "null":
                    run(["lvctl","setpass",u,pw], check=False)
        except Exception:
            pass

    # docker compose up
    if Path("/opt/marzban/docker-compose.yml").exists():
        panel("Restore", ["Docker compose up -d (marzban)…"])
        run(["bash","-lc","cd /opt/marzban && docker compose up -d --remove-orphans"], check=False)

    shutil.rmtree(TMP, ignore_errors=True)
    panel("Restore", ["Selesai."])
    pause()

def restore_from_telegram_latest():
    bot, chat = ensure_tg_config()
    pw = ensure_password()
    if not FILEID_LOG.exists():
        panel("Restore", [f"Belum ada {FILEID_LOG}"])
        pause(); return
    file_id = None
    for ln in FILEID_LOG.read_text().splitlines()[::-1]:
        if ln.startswith("File ID:"):
            file_id = ln.split(":",1)[1].strip()
            break
    if not file_id:
        panel("Restore", ["file_id terakhir tidak ditemukan di file_id.txt"])
        pause(); return

    panel("Restore", ["Ambil file_path dari Telegram…"])
    try:
        r = requests.get(f"https://api.telegram.org/bot{bot}/getFile", params={"file_id": file_id}, timeout=30)
        fp = r.json().get("result",{}).get("file_path")
    except Exception:
        fp = None
    if not fp:
        panel("Restore", ["file_path kosong."])
        pause(); return

    DL = Path(f"/root/restore-{DATE_YMD}-{DATE_HMS}.zip")
    panel("Restore", [f"Mengunduh ZIP → {DL}"])
    try:
        url = f"https://api.telegram.org/file/bot{bot}/{fp}"
        with requests.get(url, stream=True, timeout=120) as s:
            s.raise_for_status()
            with open(DL, "wb") as f:
                for chunk in s.iter_content(chunk_size=8192):
                    if chunk: f.write(chunk)
        panel("Restore", [f"ZIP diunduh: {DL}"])
    except Exception as e:
        panel("Restore", [f"Gagal download ZIP: {e}"])
        pause(); return

    restore_from_zip(str(DL))

def restore_local_zip_menu():
    panel("Restore dari ZIP Lokal", [
        "Masukkan path absolut file ZIP yang akan direstore.",
        "Contoh: /root/backup-20250101-120000.zip",
    ])
    Z = ask("Path ZIP: ").strip()
    if not Z:
        return
    restore_from_zip(Z)

# ================== CRON ==================
def set_backup_timer():
    while True:
        panel("Set Backup Timer (cron)", [
            "1) Daily",
            "2) Weekly",
            "3) Monthly",
            "4) Kembali",
        ])
        c = ask("Pilih (1-4): ").strip()
        if c == "1":
            hhmm = ask("Jam (HH:MM): ").strip()
            if ":" not in hhmm: continue
            h,m = hhmm.split(":",1)
            run(["bash","-lc", f'(crontab -l 2>/dev/null; echo "{m} {h} * * * /usr/local/sbin/lvbackup-menu --auto-backup") | crontab -'])
            panel("Cron", ["Daily diset."]); pause(); return
        elif c == "2":
            d = ask("Hari (0=Sun..6=Sat): ").strip()
            hhmm = ask("Jam (HH:MM): ").strip()
            if ":" not in hhmm: continue
            h,m = hhmm.split(":",1)
            run(["bash","-lc", f'(crontab -l 2>/dev/null; echo "{m} {h} * * {d} /usr/local/sbin/lvbackup-menu --auto-backup") | crontab -'])
            panel("Cron", ["Weekly diset."]); pause(); return
        elif c == "3":
            dom = ask("Tanggal (1-31): ").strip()
            hhmm = ask("Jam (HH:MM): ").strip()
            if ":" not in hhmm: continue
            h,m = hhmm.split(":",1)
            run(["bash","-lc", f'(crontab -l 2>/dev/null; echo "{m} {h} {dom} * * /usr/local/sbin/lvbackup-menu --auto-backup") | crontab -'])
            panel("Cron", ["Monthly diset."]); pause(); return
        elif c == "4":
            return

def remove_backup_timer():
    run(["bash","-lc", r"(crontab -l 2>/dev/null | grep -v '/usr/local/sbin/lvbackup-menu --auto-backup') | crontab -"], check=False)
    panel("Cron", ["Cron backup dibersihkan."])
    pause()

# ================== MENU ==================
def main_menu():
    while True:
        panel("Backup & Restore (Marzban + SSH-WS, Docker)", [
            "1) Backup & kirim ke Telegram",
            "2) Set Backup Timer (cron)",
            "3) Hapus Backup Timer (cron)",
            "4) Restore dari Telegram (file_id terakhir)",
            "5) Restore dari file ZIP lokal",
            "6) Keluar",
        ])
        ch = ask("Pilih (1-6): ").strip()
        if ch == "1":
            backup_and_send_telegram()
        elif ch == "2":
            set_backup_timer()
        elif ch == "3":
            remove_backup_timer()
        elif ch == "4":
            restore_from_telegram_latest()
        elif ch == "5":
            restore_local_zip_menu()
        elif ch == "6":
            os.system("clear"); sys.exit(0)

# ================== Entry ==================
def shlex_quote(s: str) -> str:
    # minimal shlex.quote (tanpa import untuk kesederhanaan)
    if not s:
        return "''"
    if re.fullmatch(r"[A-Za-z0-9@%+=:,./_-]+", s):
        return s
    return "'" + s.replace("'", "'\"'\"'") + "'"

if __name__ == "__main__":
    if len(sys.argv) > 1 and sys.argv[1] == "--auto-backup":
        backup_and_send_telegram()
        sys.exit(0)
    main_menu()
