#!/usr/bin/env python3
# menu.py - Unified launcher: Marzban (XRAY) + SSH-WS
# - Opens lingvpn_menu_mz.py (Marzban TUI)
# - Opens lvmenu (SSH-WS TUI)
# - Quick actions for Marzban CLI: up/down/restart/status/logs/update/core-update
# - Info panel ala bash: domain, ports, WARP/ufw/fail2ban status

import os, sys, shutil, subprocess, time, textwrap, re
from pathlib import Path

MARZBAN_MENU = os.environ.get("LVMZ_MENU", "/usr/local/sbin/lingvpn_menu_mz.py")
SSHWS_MENU   = os.environ.get("LV_SSH_MENU", "/usr/local/sbin/lvmenu")
DOMAIN_FILE  = "/root/domain"

# =============== helpers ===============
def run_cmd_show(cmd: str) -> int:
    print(f"$ {cmd}")
    p = subprocess.run(cmd, shell=True, text=True)
    return p.returncode

def read_domain():
    try:
        return Path(DOMAIN_FILE).read_text().strip()
    except Exception:
        return "example.id"

def run(cmd, shell=False, capture=False):
    if capture:
        return subprocess.run(cmd, shell=shell, text=True, capture_output=True)
    return subprocess.call(cmd, shell=shell)

def check_port_local_40000():
    # wireproxy WARP check 127.0.0.1:40000
    try:
        r = run("ss -ltn 2>/dev/null | grep -E '127\\.0\\.0\\.1:40000\\s'", shell=True, capture=True)
        if r and r.stdout.strip():
            return True
        r = run("netstat -ntlp 2>/dev/null | grep -E '127\\.0\\.0\\.1:40000'", shell=True, capture=True)
        return bool(r and r.stdout.strip())
    except Exception:
        return False

def systemd_active(unit):
    r = run(["systemctl","is-active","--quiet",unit])
    return (r == 0)

def colorize(s, c):
    if not sys.stdout.isatty(): return s
    C = {
        "red":"\033[0;31m","green":"\033[0;32m","yellow":"\033[0;33m",
        "blue":"\033[0;34m","cyan":"\033[0;36m","reset":"\033[0m"
    }
    return f"{C.get(c,'')}{s}{C['reset']}"

def do_backup_restore():
    rc = run_cmd_show("lvbackup-menu")
    if rc != 0:
        print("Gagal menjalankan lvbackup-menu (pastikan terpasang & executable).")

# =============== ANSI panel (same style) ===============
_ANSI_RE = re.compile(r"\x1b\[[0-9;]*m")
def _ansi(name):
    if not sys.stdout.isatty(): return ""
    return {
        "reset":"\033[0m","muted":"\033[38;5;245m",
        "title":"\033[38;5;111m","num":"\033[38;5;81m","accent":"\033[38;5;45m",
    }.get(name,"")
def _vislen(s): return len(_ANSI_RE.sub("", s or ""))
def _pad_to(s,w): return s + (" " * max(0, w-_vislen(s)))
def _hr(w): return "─" * max(0,w)
def _menu_line(num,label,width_num=2,width_label=44):
    n=f"{_ansi('num')}{str(num).rjust(width_num)}{_ansi('reset')})"
    parts=textwrap.wrap(label,width_label) or [label]
    out=[f"{n}  {parts[0]}"]
    for ln in parts[1:]:
        out.append(" " * (width_num+3) + ln)
    return out
def panel_header(title):
    cols=shutil.get_terminal_size(fallback=(100,28)).columns
    panel_w=min(100,max(70,cols-4))
    top="┌"+_hr(panel_w)+"┐"; sep="├"+_hr(panel_w)+"┤"; bot="└"+_hr(panel_w)+"┘"
    return panel_w, top, sep, bot

def safe_input(prompt):
    try:
        return input(prompt)
    except (KeyboardInterrupt, EOFError):
        print("\n(dibatalkan)")
        return None

def pause():
    _ = safe_input("\n[Enter] untuk kembali...")

# =============== info block ===============
def show_info_block():
    domain = read_domain()
    warp_ok = check_port_local_40000()
    ufw = systemd_active("ufw")
    f2b = systemd_active("fail2ban")

    cols = shutil.get_terminal_size(fallback=(100,28)).columns
    panel_w, top, sep, bot = panel_header("Info")

    print(top)
    title = f"{_ansi('title')}LingVPN XRAY + SSH-WS Menu{_ansi('reset')}"
    print("│ " + _pad_to(title, panel_w-1) + "│")
    print(sep)

    def line(s):
        # pastikan tidak melebihi lebar panel (hindari wrap)
        if _vislen(s) > panel_w - 1:
            s = s[:panel_w - 4] + "…"
        print("│ " + _pad_to(s, panel_w-1) + "│")

    # gunakan ASCII checkbox agar aman di semua terminal
    cb = lambda ok: "[x]" if ok else "[ ]"

    line(f"Domain                : {domain}")
    line(f"Dashboard Marzban GUI : https://{domain}/dashboard")
    line(f"Ports (TLS)           : 443, 8443, 8880")
    line(f"Ports (nTLS/HTTP)     : 80, 2082, 2083, 3128, 8080")
    line(f"VLESS Reality TCP+gRPC: 443")
    line(f"WARP Wireproxy        : {cb(warp_ok)} 127.0.0.1:40000")
    line(f"Firewall (ufw)        : {cb(ufw)} ON")
    line(f"Fail2ban              : {cb(f2b)} ON")
    line(f"Timezone              : Asia/Jakarta (+7)")
    print(bot)

# =============== actions ===============
def open_marzban_menu():
    if not os.path.exists(MARZBAN_MENU):
        print(f"Marzban menu tidak ditemukan: {MARZBAN_MENU}")
        return
    # JANGAN execv — cukup panggil dan tunggu selesai
    if MARZBAN_MENU.endswith(".py"):
        subprocess.call([sys.executable, MARZBAN_MENU])
    else:
        subprocess.call([MARZBAN_MENU])
    # setelah return, kita lanjut ke menu utama lagi

def open_sshws_menu():
    path = SSHWS_MENU
    if not os.path.exists(path):
        alt = "/usr/local/sbin/sshws_menu.py"
        if os.path.exists(alt):
            path = alt
        else:
            print(f"SSH-WS menu tidak ditemukan: {SSHWS_MENU}")
            return
    if path.endswith(".py"):
        subprocess.call([sys.executable, path])
    else:
        subprocess.call([path])

def marzban_cmd(sub):
    # direct marzban CLI
    rc = run(["marzban", sub])
    if rc != 0:
        print(f"Perintah 'marzban {sub}' gagal (rc={rc}).")
    pause()

def marzban_logs():
    print("Menjalankan: marzban logs (Ctrl+C untuk keluar)")
    try:
        run(["marzban","logs"])
    except KeyboardInterrupt:
        pass

def marzban_status():
    run(["marzban","status"])
    pause()

def marzban_core_update():
    rc = run(["marzban","core-update"])
    if rc != 0:
        print("core-update gagal.")
    pause()

# =============== submenus ===============
def submenu_marzban():
    items = [
        ("1","Up (start)",           lambda: marzban_cmd("up")),
        ("2","Down (stop)",          lambda: marzban_cmd("down")),
        ("3","Restart",              lambda: marzban_cmd("restart")),
        ("4","Status",               marzban_status),
        ("5","Logs (follow)",        marzban_logs),
        ("6","Update (pull images)", lambda: marzban_cmd("update")),
        ("7","Core-update (Xray)",   marzban_core_update),
        ("0","Kembali",              None),
    ]
    while True:
        os.system("clear")
        panel_w, top, sep, bot = panel_header("Marzban Controls")
        print(top)
        print("│ " + _pad_to(f"{_ansi('title')}Marzban Controls{_ansi('reset')}", panel_w-1) + "│")
        print(sep)
        width_num = 2; width_label = panel_w - (width_num + 5)
        for k, label, _ in items:
            for ln in _menu_line(k,label,width_num,width_label):
                print("│ " + _pad_to(ln, panel_w-1) + "│")
        print(sep)
        tip=f"{_ansi('muted')}Ctrl+C: batal/ulang · Enter kosong: ulang menu{_ansi('reset')}"
        print("│ " + _pad_to(tip, panel_w-1) + "│")
        print(bot)

        ch = safe_input(f"{_ansi('accent')}Pilih{_ansi('reset')}: ")
        if ch is None:  # ulang
            continue
        ch = ch.strip()
        matched=False
        for k, _, fn in items:
            if ch == k:
                matched=True
                if k=="0": return
                try:
                    fn()
                except KeyboardInterrupt:
                    pass
                break
        if not matched and ch!="":
            print("Pilihan tidak dikenal."); time.sleep(0.8)

# =============== main menu ===============
def main():
    if os.geteuid()!=0:
        print("Jalankan sebagai root."); sys.exit(1)

    while True:
        os.system("clear")
        show_info_block()

        items = [
            ("1","XRAY / Marzban Menu", open_marzban_menu),
            ("2","SSH-WS Menu",         open_sshws_menu),
            ("3","Marzban Controls",    submenu_marzban),
            ("4","Backup/Restore",      do_backup_restore),
            ("0","Keluar",              None),
        ]
        panel_w, top, sep, bot = panel_header("Main Menu")
        print(top)
        print("│ " + _pad_to(f"{_ansi('title')}Main Menu{_ansi('reset')}", panel_w-1) + "│")
        print(sep)
        width_num = 2; width_label = panel_w - (width_num + 5)
        for k, label, _ in items:
            for ln in _menu_line(k,label,width_num,width_label):
                print("│ " + _pad_to(ln, panel_w-1) + "│")
        print(sep)
        tip=f"{_ansi('muted')}Ctrl+C: batal/ulang · Enter kosong: ulang menu{_ansi('reset')}"
        print("│ " + _pad_to(tip, panel_w-1) + "│")
        print(bot)

        ch = safe_input(f"{_ansi('accent')}Pilih{_ansi('reset')}: ")
        if ch is None:
            continue
        ch = ch.strip()
        matched=False
        for k, _, fn in items:
            if ch == k:
                matched=True
                if k=="0":
                 print("bye."); return
                try:
                    fn()
                except KeyboardInterrupt:
                    pass
                # note: open_* functions execv & replace process, so return only if failed
                break
        if not matched and ch!="":
            print("Pilihan tidak dikenal."); time.sleep(0.8)

if __name__ == "__main__":
    main()
