#!/usr/bin/env bash
# LingVPN + Marzban - Backup & Restore (Telegram) — DOCKER NGINX VERSION
# - Backup:
#     * /opt/marzban (docker-compose.yml, nginx conf, dll)
#     * /var/lib/marzban/{db.sqlite3,xray_config.json}
#     * /var/www/html
#     * /etc/lingvpn (db.json, sub.secret)
#     * /usr/local/sbin/{wsproxy.py,lvctl,lvmenu,lvip,lvsubd.py}
#     * /etc/systemd/system/lvsubd.service
#     * /var/www/html/lvsub/index.html (subscription page)
#     * /var/lib/vnstat/vnstat.db
#     * /root/{telegram_config.conf,file_id.txt}
# - Restore: copy balik & (re)start lvsubd + docker compose up -d
set -euo pipefail

CFG_TG="/root/telegram_config.conf"
PASS_FILE="/root/passbackup"
FILEID_LOG="/root/file_id.txt"
LOG_BACKUP="/root/log-backup.txt"
NAME_FILE="/root/nama"
DATE_YMD=$(date +"%Y%m%d"); DATE_HMS=$(date +"%H%M%S"); DATE_DASH=$(date +"%m-%d-%Y")
green='\e[32;1m'; red='\e[31;1m'; nc='\e[0m'
say(){ echo -e "$*"; } ; ok(){ say "${green}[OK]${nc} $*"; } ; err(){ say "${red}[ERR]${nc} $*"; }
need(){ command -v "$1" >/dev/null 2>&1 || { err "butuh '$1'"; exit 1; }; }
need curl; need jq; need zip; need unzip

ensure_tg_config(){
  if [[ ! -f "$CFG_TG" ]]; then
    say "Bot token & chat ID belum ada."
    read -rp "Telegram bot token: " botToken
    read -rp "Telegram chat ID : " chatId
    echo "botToken=$botToken" > "$CFG_TG"
    echo "chatId=$chatId"   >> "$CFG_TG"
    ok "Disimpan: $CFG_TG"; exit 0
  fi
  # shellcheck disable=SC1090
  source "$CFG_TG"
  [[ -n "${botToken:-}" && -n "${chatId:-}" ]] || { err "botToken/chatId kosong"; exit 1; }
}
ensure_password(){ [[ -s "$PASS_FILE" ]] || { err "Buat password zip di $PASS_FILE"; exit 1; } }
get_name_base(){ [[ -s "$NAME_FILE" ]] && cat "$NAME_FILE" || echo "lingvpn"; }

print_header(){
  clear
  cat <<'HDR'
**************************************************
*                                                *
*   Backup & Restore (Marzban + SSH-WS, Docker)  *
*                                                *
**************************************************
HDR
}
pause(){ read -rp "Tekan ENTER untuk lanjut… " _; }

# ---------- BACKUP: hanya /opt/marzban utk docker+nginx ----------
backup_build_tree(){
  local BKROOT="/root/backup"; rm -rf "$BKROOT"; mkdir -p "$BKROOT/backup"
  # Docker stack (nginx + compose) — CUKUP INI
  [[ -e /opt/marzban                  ]] && cp -a /opt/marzban                 "$BKROOT/backup/"
  # Data marzban
  [[ -e /var/lib/marzban/db.sqlite3   ]] && cp -a /var/lib/marzban/db.sqlite3  "$BKROOT/backup/"
  [[ -e /var/lib/marzban/xray_config.json ]] && cp -a /var/lib/marzban/xray_config.json "$BKROOT/backup/"
  # Web umum & lvsub page
  [[ -e /var/www/html                 ]] && cp -a /var/www/html                 "$BKROOT/backup/"
  [[ -e /var/www/html/lvsub/index.html ]] && { mkdir -p "$BKROOT/backup/var_www_lvsub"; cp -a /var/www/html/lvsub/index.html "$BKROOT/backup/var_www_lvsub/"; }
  # LingVPN (SSH-WS)
  [[ -e /etc/lingvpn                  ]] && cp -a /etc/lingvpn                  "$BKROOT/backup/"
  for f in /usr/local/sbin/wsproxy.py /usr/local/sbin/lvctl /usr/local/sbin/lvmenu /usr/local/sbin/lvbackup-menu /usr/local/sbin/lvsubd.py; do
    [[ -e "$f" ]] && cp -a "$f" "$BKROOT/backup/"
  done
  [[ -e /etc/systemd/system/lvsubd.service ]] && cp -a /etc/systemd/system/lvsubd.service "$BKROOT/backup/"
  # VNStat + Telegram meta
  [[ -e /var/lib/vnstat/vnstat.db     ]] && cp -a /var/lib/vnstat/vnstat.db     "$BKROOT/backup/"
  [[ -e $CFG_TG                       ]] && cp -a "$CFG_TG"                     "$BKROOT/backup/telegram_config.conf"
  [[ -e $FILEID_LOG                   ]] && cp -a "$FILEID_LOG"                 "$BKROOT/backup/file_id.txt"
  echo "$BKROOT"
}

backup_and_send_telegram(){
  print_header; ensure_tg_config; ensure_password
  local NAME BASE ts zipfile BKROOT
  NAME=$(get_name_base); BASE="${NAME}_${DATE_HMS}_${DATE_YMD}"; ts=$(date +"%T")
  say "Memulai backup…"; BKROOT=$(backup_build_tree)
  pushd "$BKROOT" >/dev/null
  zipfile="/root/${BASE}.zip"
  say "[*] ZIP (password)…"
  (cd backup && zip -rP "$(cat "$PASS_FILE")" "$zipfile" . >/dev/null)
  popd >/dev/null
  [[ -f "$zipfile" ]] || { err "Gagal bikin ZIP"; exit 1; }
  say "[*] Kirim Telegram…"
  response=$(curl -fsS -F chat_id="$chatId" -F document=@"$zipfile" "https://api.telegram.org/bot${botToken}/sendDocument")
  file_id=$(echo "$response" | jq -r '.result.document.file_id // empty')
  if [[ -n "$file_id" ]]; then
    ok "Terkirim. file_id: $file_id"
    { echo "### ${NAME}_${DATE_DASH}"; echo "File ID: $file_id"; } >> "$FILEID_LOG"
    echo "Server backup pada ${DATE_DASH} ${ts}" >> "$LOG_BACKUP"
  else
    err "Gagal kirim. Response:"; echo "$response"
  fi
  rm -rf /root/backup || true; rm -f "$zipfile" || true
  pause
}

# ---------- RESTORE ----------
restore_from_zip(){
  local ZIP="$1"; [[ -s "$ZIP" ]] || { err "ZIP tidak ditemukan: $ZIP"; return; }
  local TMP="/root/restore-work"; rm -rf "$TMP"; mkdir -p "$TMP"

  say "[*] Unzip…"
  unzip -P "$(cat "$PASS_FILE")" "$ZIP" -d "$TMP" >/dev/null || { err "Unzip gagal (password?)"; rm -rf "$TMP"; return; }

  # Stop layanan dulu
  systemctl stop lvsubd 2>/dev/null || true
  systemctl stop wsproxy 2>/dev/null || true

  # Stop stack marzban (kalau ada)
  if [[ -f /opt/marzban/docker-compose.yml ]]; then
    say "[*] Docker compose down (marzban)…"
    (cd /opt/marzban && docker compose down) || true
  fi

  pushd "$TMP" >/dev/null

  # Pulihkan /opt/marzban (satu paket docker+nginx)
  [[ -e ./opt/marzban ]] && cp -a ./opt/marzban /opt/

  # Data marzban
  [[ -e ./var/lib/marzban/db.sqlite3       ]] && cp -a ./var/lib/marzban/db.sqlite3 /var/lib/marzban/db.sqlite3
  [[ -e ./var/lib/marzban/xray_config.json ]] && cp -a ./var/lib/marzban/xray_config.json /var/lib/marzban/xray_config.json

  # Web root (termasuk lvsub)
  [[ -e ./var/www/html        ]] && cp -a ./var/www/html /var/www/
  [[ -e ./var_www_lvsub/index.html ]] && { mkdir -p /var/www/html/lvsub; cp -a ./var_www_lvsub/index.html /var/www/html/lvsub/index.html; }

  # LingVPN files
  [[ -e ./etc/lingvpn            ]] && cp -a ./etc/lingvpn /etc/
  for f in wsproxy.py lvctl lvmenu vsubd.py; do
    [[ -e "./$f" ]] && cp -a "./$f" "/usr/local/sbin/$f"
  done
  [[ -e ./etc/systemd/system/lvsubd.service ]] && cp -a ./etc/systemd/system/lvsubd.service /etc/systemd/system/lvsubd.service

  # vnstat + telegram meta
  [[ -e ./var/lib/vnstat/vnstat.db ]] && cp -a ./var/lib/vnstat/vnstat.db /var/lib/vnstat/vnstat.db
  [[ -e ./telegram_config.conf     ]] && cp -a ./telegram_config.conf "$CFG_TG"
  [[ -e ./file_id.txt              ]] && cp -a ./file_id.txt          "$FILEID_LOG"

  popd >/dev/null

  # Executable bit
  chmod +x /usr/local/sbin/wsproxy.py  2>/dev/null || true
  chmod +x /usr/local/sbin/lvctl       2>/dev/null || true
  chmod +x /usr/local/sbin/lvmenu      2>/dev/null || true
  chmod +x /usr/local/sbin/lvsubd.py   2>/dev/null || true

  # Reload & start systemd services
  systemctl daemon-reload
  systemctl restart lvsubd 2>/dev/null || true
  systemctl restart wsproxy 2>/dev/null || true

  # Re-provision user OS dari DB (prefer jq; fallback python -c one-liner)
  if [[ -s /etc/lingvpn/db.json ]]; then
    echo "[*] Re-provision SSH-WS users dari /etc/lingvpn/db.json …"
    if command -v jq >/dev/null 2>&1; then
      jq -r '.users | keys[]' /etc/lingvpn/db.json | while read -r U; do
        PW=$(jq -r --arg u "$U" '.users[$u].password' /etc/lingvpn/db.json)
        if [[ -n "$PW" && "$PW" != "null" ]]; then
          lvctl setpass "$U" "$PW" || true
        fi
      done
    else
      python3 -c 'import json,subprocess; 
d=json.load(open("/etc/lingvpn/db.json"));
for u,inf in d.get("users",{}).items():
  pw=str(inf.get("password") or "");
  if pw and pw!="null":
    subprocess.run(["lvctl","setpass",u,pw], check=False)'
    fi
  fi

  # Start kembali docker compose marzban
  if [[ -f /opt/marzban/docker-compose.yml ]]; then
    say "[*] Docker compose up -d (marzban)…"
    (cd /opt/marzban && docker compose up -d --remove-orphans) || true
  fi

  rm -rf "$TMP"
  ok "Restore selesai."
  pause
}

restore_from_telegram_latest(){
  print_header; ensure_tg_config; ensure_password
  [[ -s "$FILEID_LOG" ]] || { err "Belum ada $FILEID_LOG"; pause; return; }
  FILE_ID=$(grep '^File ID:' "$FILEID_LOG" | awk '{print $3}' | tail -n 1)
  [[ -n "$FILE_ID" ]] || { err "file_id tidak ditemukan"; pause; return; }
  say "[*] getFile…"
  FILE_PATH=$(curl -fsS "https://api.telegram.org/bot${botToken}/getFile?file_id=${FILE_ID}" | jq -r '.result.file_path // empty')
  [[ -n "$FILE_PATH" ]] || { err "file_path kosong"; pause; return; }
  DL="/root/restore-${DATE_YMD}-${DATE_HMS}.zip"
  say "[*] download ZIP…"; curl -fsS -o "$DL" "https://api.telegram.org/file/bot${botToken}/${FILE_PATH}"
  ok "ZIP diunduh: $DL"
  restore_from_zip "$DL"
}

restore_local_zip_menu(){ print_header; read -rp "Path ZIP lokal: " Z; [[ -n "$Z" ]] || { err "dibatalkan"; pause; return; }; restore_from_zip "$Z"; }

# ---------- CRON ----------
set_backup_timer(){
  print_header
  cat <<MENU
Set Backup Timer:
1) Daily
2) Weekly
3) Monthly
4) Kembali
MENU
  read -rp "Pilih (1-4): " c
  case "$c" in
    1) read -rp "Jam (HH:MM): " hhmm; h=${hhmm%:*}; m=${hhmm#*:}
       (crontab -l 2>/dev/null; echo "$m $h * * * /usr/local/sbin/lvbackup-menu --auto-backup") | crontab - ;;
    2) read -rp "Hari (0-6): " d; read -rp "Jam (HH:MM): " hhmm; h=${hhmm%:*}; m=${hhmm#*:}
       (crontab -l 2>/dev/null; echo "$m $h * * $d /usr/local/sbin/lvbackup-menu --auto-backup") | crontab - ;;
    3) read -rp "Tanggal (1-31): " dom; read -rp "Jam (HH:MM): " hhmm; h=${hhmm%:*}; m=${hhmm#*:}
       (crontab -l 2>/dev/null; echo "$m $h $dom * * /usr/local/sbin/lvbackup-menu --auto-backup") | crontab - ;;
    *) ;;
  esac
  ok "Cron diset."; pause
}
remove_backup_timer(){ print_header; crontab -l 2>/dev/null | grep -v "/usr/local/sbin/lvbackup-menu --auto-backup" | crontab - || true; ok "Cron dibersihkan."; pause; }

# ---------- MENU ----------
main_menu(){
  while :; do
    print_header
    cat <<MENU
Main Menu:
1) Backup & kirim ke Telegram
2) Set Backup Timer (cron)
3) Hapus Backup Timer (cron)
4) Restore dari Telegram (file_id terakhir)
5) Restore dari file ZIP lokal
6) Keluar
MENU
    read -rp "Pilih (1-6): " ch
    case "$ch" in
      1) backup_and_send_telegram ;;
      2) set_backup_timer ;;
      3) remove_backup_timer ;;
      4) restore_from_telegram_latest ;;
      5) restore_local_zip_menu ;;
      6) exit 0 ;;
      *) say "Pilihan tidak valid." ; sleep 1 ;;
    esac
  done
}

# ---------- ENTRY ----------
if [[ "${1:-}" == "--auto-backup" ]]; then backup_and_send_telegram; exit 0; fi
main_menu
