#!/usr/bin/env python3
# lvctl - LingVPN control (proxy-accounting only; no iptables/nft)
# - Accounting & limits handled by wsproxy.py and lingvpn-shell.
# - This tool manages OS users + /etc/lingvpn/db.json, and can kill sessions.
# - No firewall manipulation here.

import json, sys, subprocess, time, os, datetime, shutil

DB_PATH    = "/etc/lingvpn/db.json"
SHELL_PATH = "/usr/local/sbin/lingvpn-shell"
GiB        = 1024**3

# -------------------- base utils --------------------
def epoch() -> int:
    return int(time.time())

def iso_now() -> str:
    return datetime.datetime.utcnow().replace(microsecond=0).isoformat() + "Z"

def load_db() -> dict:
    if not os.path.exists(DB_PATH):
        return {"version":1, "last_update": iso_now(), "netns":"dbns0", "users":{}}
    with open(DB_PATH,"r") as f:
        try:
            return json.load(f)
        except Exception:
            return {"version":1, "last_update": iso_now(), "netns":"dbns0", "users":{}}

def save_db(db: dict):
    db["last_update"] = iso_now()
    os.makedirs(os.path.dirname(DB_PATH), exist_ok=True)
    tmp = DB_PATH + ".tmp"
    with open(tmp,"w") as f:
        json.dump(db, f, indent=2, sort_keys=True)
    os.replace(tmp, DB_PATH)

def req_root():
    if os.geteuid() != 0:
        print("Run as root", file=sys.stderr); sys.exit(1)

def run(cmd_list):
    return subprocess.run(cmd_list, check=False, capture_output=True, text=True)

# -------------------- bytes & time helpers --------------------
def fmt_bytes_auto(b):
    try: b = int(b)
    except: b = 0
    units = ["B","KB","MB","GB","TB","PB"]
    val = float(b); i = 0
    while val >= 1024.0 and i < len(units)-1:
        val /= 1024.0; i += 1
    return f"{int(val)} {units[i]}" if i == 0 else f"{val:.2f} {units[i]}"

def parse_quota_gb_to_bytes(s):
    s = (s or "").strip().lower()
    if s in ("0","0g","0gb","0gib","unlimited","∞","inf","infinite"):
        return 0
    num = ""
    for ch in s:
        if ch.isdigit() or ch in ".": num += ch
        else: break
    if not num: return 0
    return int(float(num) * GiB)

def fmt_expiry(exp_epoch):
    try: exp = int(exp_epoch)
    except: exp = 0
    if exp <= 0: return "Never"
    dt = datetime.datetime.utcfromtimestamp(exp)
    delta = dt - datetime.datetime.utcnow()
    days_left = max(0, delta.days)
    return f"{dt.strftime('%Y-%m-%d %H:%M:%SZ')}  ({days_left}d)"

def fmt_duration_secs(secs):
    secs = int(max(0, secs))
    if secs <= 0: return "0s"
    m, s = divmod(secs, 60)
    h, m = divmod(m, 60)
    d, h = divmod(h, 24)
    parts = []
    if d: parts.append(f"{d}d")
    if h: parts.append(f"{h}h")
    if m: parts.append(f"{m}m")
    if s or not parts: parts.append(f"{s}s")
    return " ".join(parts)

# -------------------- OS user helpers --------------------
def ensure_shell_registered():
    try:
        with open("/etc/shells","r+",encoding="utf-8") as f:
            shells = [ln.strip() for ln in f.readlines()]
            if SHELL_PATH not in shells:
                f.write("\n"+SHELL_PATH+"\n")
    except FileNotFoundError:
        with open("/etc/shells","w",encoding="utf-8") as f:
            f.write("/bin/sh\n/bin/bash\n"+SHELL_PATH+"\n")

def user_exists(u):
    return run(["id","-u",u]).returncode == 0

def ensure_user_sys(u, pw):
    ensure_shell_registered()
    if not user_exists(u):
        subprocess.run(["useradd","-m","-s",SHELL_PATH,u], check=True)
    else:
        subprocess.run(["usermod","-s",SHELL_PATH,u], check=True)
    if pw:
        p = subprocess.Popen(["chpasswd","-c","SHA512"], stdin=subprocess.PIPE, text=True)
        p.communicate(f"{u}:{pw}")
        if p.returncode != 0:
            print("chpasswd failed", file=sys.stderr); sys.exit(1)
    subprocess.run(["passwd","-u",u], stdout=subprocess.DEVNULL, stderr=subprocess.DEVNULL)
    subprocess.run(["chage","-E","-1",u], stdout=subprocess.DEVNULL, stderr=subprocess.DEVNULL)

def hard_kill_user(u):
    # Try logind (terminates cgroup & sessions)
    if shutil.which("loginctl"):
        try:
            subprocess.run(["loginctl","terminate-user",u], stdout=subprocess.DEVNULL, stderr=subprocess.DEVNULL)
            uid = int(subprocess.check_output(["id","-u",u]).decode().strip())
            subprocess.run(["loginctl","kill-user",str(uid),"KILL"], stdout=subprocess.DEVNULL, stderr=subprocess.DEVNULL)
        except Exception:
            pass
    # Fallback: kill all processes by UID
    subprocess.run(["pkill","-9","-u",u], stdout=subprocess.DEVNULL, stderr=subprocess.DEVNULL)

def delete_user_sys(u):
    hard_kill_user(u)
    if user_exists(u):
        subprocess.run(["userdel","-r",u], stdout=subprocess.DEVNULL, stderr=subprocess.DEVNULL)

# -------------------- status/labels --------------------
def calc_status(info):
    now = epoch()
    if not info.get("enabled", True): return "Disabled"
    if int(info.get("lock_until",0)) > now: return "Locked"
    exp = int(info.get("expire_at",0))
    if exp>0 and now>exp: return "Expired"
    q = int(info.get("quota_bytes",0))
    used = int(info.get("used_bytes",0))
    if q>0 and used>=q: return "Over-quota"
    return "Active"

def limit_label(qbytes):
    q = int(qbytes)
    return "Unlimited" if q == 0 else fmt_bytes_auto(q)

def reset_label(s):
    return {"daily":"Daily","weekly":"Weekly","monthly":"Monthly","none":"-"}\
           .get((s or "").lower(), s or "-")

# -------------------- table rendering --------------------
def print_table(rows):
    widths = [max(len(str(cell)) for cell in col) for col in zip(*rows)]
    def line(char='-'):
        print(' '.join(char * w for w in widths))
    hdr = rows[0]
    print(' '.join(str(h).ljust(w) for h,w in zip(hdr,widths)))
    line('-')
    for r in rows[1:]:
        print(' '.join(str(c).ljust(w) for c,w in zip(r,widths)))

# -------------------- commands --------------------
def cmd_add(args):
    # lvctl add <user> <password> [days=30] [quota_gb=0] [max_devices=3] [reset=daily|weekly|monthly|none]
    if len(args) < 3:
        print("usage: lvctl add <user> <password> [days=30] [quota_gb=0] [max_devices=3] [reset=daily|weekly|monthly|none]")
        sys.exit(1)
    user, pw = args[1], args[2]
    days      = int(args[3]) if len(args)>3 else 30
    quota_gb  = args[4] if len(args)>4 else "0"
    max_dev   = int(args[5]) if len(args)>5 else 3
    reset_str = args[6] if len(args)>6 else "monthly"

    reset_seconds_map = {"daily":86400,"weekly":604800,"monthly":2592000,"none":0}
    reset_seconds = reset_seconds_map.get(reset_str, 2592000)

    db = load_db()
    ensure_user_sys(user, pw)

    now = epoch()
    exp = now + days*86400 if days>0 else 0
    qbytes = parse_quota_gb_to_bytes(quota_gb)

    users = db.setdefault("users",{})
    rec = users.get(user, {})
    if not rec:
        rec = {
            "password": pw,
            "quota_bytes": qbytes,
            "used_bytes": 0,
            "cum_used_bytes": 0,
            "enabled": True,
            "created_at": iso_now(),
            "last_update": iso_now(),
            "expire_at": exp,
            "reset_every_seconds": reset_seconds,
            "last_reset_at": now,
            "reset_strategy": reset_str,
            "last_reset_key": datetime.datetime.utcnow().strftime("%Y-%m") if reset_str=="monthly" else "",
            "max_devices": max_dev,
            "lock_until": 0
        }
    else:
        rec["password"] = pw
        rec["quota_bytes"] = qbytes
        rec["max_devices"] = max_dev
        rec["expire_at"] = exp
        rec["reset_every_seconds"] = reset_seconds
        rec["reset_strategy"] = reset_str
        rec.setdefault("used_bytes", 0)
        rec.setdefault("cum_used_bytes", 0)
        rec["last_update"] = iso_now()
    users[user] = rec
    save_db(db)
    print(f"OK add {user}")

def cmd_setpass(args):
    if len(args) < 3:
        print("usage: lvctl setpass <user> <password>"); sys.exit(1)
    user, pw = args[1], args[2]
    db = load_db()
    if user not in db.get("users", {}): print("no such user"); sys.exit(1)
    ensure_user_sys(user, pw)
    db["users"][user]["password"] = pw
    db["users"][user]["last_update"] = iso_now()
    save_db(db)
    print("OK")

def cmd_setquota(args):
    if len(args) < 3:
        print("usage: lvctl setquota <user> <quota_gb>"); sys.exit(1)
    user, qgb = args[1], args[2]
    db = load_db()
    if user not in db.get("users", {}): print("no such user"); sys.exit(1)
    qbytes = parse_quota_gb_to_bytes(qgb)
    db["users"][user]["quota_bytes"] = qbytes
    db["users"][user]["last_update"] = iso_now()
    save_db(db)
    print("OK")

def cmd_enable(args, val=True):
    if len(args) < 2:
        print(f"usage: lvctl {'enable' if val else 'disable'} <user>"); sys.exit(1)
    user = args[1]
    db = load_db()
    if user not in db.get("users", {}): print("no such user"); sys.exit(1)
    db["users"][user]["enabled"] = bool(val)
    db["users"][user]["last_update"] = iso_now()
    save_db(db)
    if not val:
        hard_kill_user(user)
        print("OK (disabled & kicked)")
    else:
        print("OK (enabled)")

def cmd_lock(args):
    if len(args) < 3:
        print("usage: lvctl lock <user> <seconds>"); sys.exit(1)
    user, secs = args[1], int(args[2])
    db = load_db()
    if user not in db.get("users", {}): print("no such user"); sys.exit(1)
    until = epoch() + secs
    db["users"][user]["lock_until"] = until
    db["users"][user]["last_update"] = iso_now()
    save_db(db)
    hard_kill_user(user)
    print(f"OK (locked {secs}s & kicked)")

def cmd_delete(args):
    if len(args) < 2:
        print("usage: lvctl delete <user>"); sys.exit(1)
    user = args[1]
    delete_user_sys(user)
    db = load_db()
    db.get("users",{}).pop(user, None)
    save_db(db)
    print("OK delete", user)

def cmd_info(args):
    if len(args) < 2:
        print("usage: lvctl info <user> [--json] [--show-pass]"); sys.exit(1)
    user      = args[1]
    as_json   = ("--json" in args[2:])
    show_pass = ("--show-pass" in args[2:])

    db = load_db()
    info = db.get("users", {}).get(user)
    if not info:
        print("no such user"); sys.exit(1)

    if as_json:
        print(json.dumps(info, indent=2, sort_keys=True)); return

    now     = epoch()
    status  = calc_status(info)
    enabled = bool(info.get("enabled", True))
    lock_u  = int(info.get("lock_until", 0))
    lock_rem= fmt_duration_secs(lock_u - now) if lock_u > now else "-"

    used    = int(info.get("used_bytes", 0))
    cum     = int(info.get("cum_used_bytes", 0))
    limit_b = int(info.get("quota_bytes", 0))
    limit_h = limit_label(limit_b)
    used_h  = fmt_bytes_auto(used)
    cum_h   = fmt_bytes_auto(cum)

    reset_s = info.get("reset_strategy","-")
    reset_h = reset_label(reset_s)
    exp_e   = int(info.get("expire_at", 0))
    exp_h   = fmt_expiry(exp_e)

    max_dev = int(info.get("max_devices", 0))
    created = info.get("created_at","-")
    updated = info.get("last_update","-")

    pw = info.get("password","")
    pw_line = pw if show_pass else "(hidden)  # gunakan --show-pass untuk melihat"

    print()
    print(f"User            : {user}")
    print(f"Status          : {status} {'(disabled)' if not enabled else ''}".rstrip())
    print(f"Password        : {pw_line}")
    print(f"Max Devices     : {max_dev}")
    print(f"Lock Remaining  : {lock_rem}")
    print(f"Quota Limit     : {limit_h}")
    print(f"Used (window)   : {used_h}")
    print(f"Total Used      : {cum_h}")
    print(f"Reset Strategy  : {reset_h}")
    print(f"Expires At      : {exp_h}")
    print(f"Created At      : {created}")
    print(f"Last Update     : {updated}")
    if info.get("last_ip") or info.get("last_connect_at"):
        print(f"Last IP         : {info.get('last_ip','-')}")
        print(f"Last Connect At : {info.get('last_connect_at','-')}")
    print()

def cmd_collect(args):
    db = load_db()
    changed=False
    for _,info in db.get("users",{}).items():
        if "used_bytes" not in info: info["used_bytes"]=0; changed=True
        if "cum_used_bytes" not in info: info["cum_used_bytes"]=0; changed=True
        if "last_update" not in info: info["last_update"]=iso_now(); changed=True
        if "enabled" not in info: info["enabled"]=True; changed=True
    if changed: save_db(db)
    print("OK collect")

def cmd_enforce(args):
    db = load_db()
    now = epoch()
    kicked=[]
    for u,info in db.get("users",{}).items():
        disabled = not info.get("enabled",True)
        locked   = int(info.get("lock_until",0)) > now
        expired  = (int(info.get("expire_at",0))>0 and now>int(info["expire_at"]))
        overq    = (int(info.get("quota_bytes",0))>0 and int(info.get("used_bytes",0)) >= int(info["quota_bytes"]))
        if disabled or locked or expired or overq:
            hard_kill_user(u)
            kicked.append(u)
    print("Enforced: " + (", ".join(kicked) if kicked else "(none)"))

def cmd_list(args):
    db = load_db()
    now = epoch()
    users = db.get("users", {})
    rows = [["No.","Username","Status","Penggunaan Data","Total Penggunaan Data","Data Limit","Reset Strategy","Max Dev","Expires At","Lock Remaining"]]
    n = 0
    for u in sorted(users.keys()):
        info = users[u]
        n += 1
        status = calc_status(info)
        used   = fmt_bytes_auto(info.get("used_bytes",0))
        cum    = fmt_bytes_auto(info.get("cum_used_bytes",0))
        limit  = limit_label(info.get("quota_bytes",0))
        reset  = reset_label(info.get("reset_strategy","-"))
        exp    = fmt_expiry(info.get("expire_at",0))
        lock_until = int(info.get("lock_until",0))
        lock_rem = fmt_duration_secs(lock_until - now) if lock_until > now else "-"
        rows.append([str(n), u, status, used, cum, limit, reset, str(info.get("max_devices",0)), exp, lock_rem])
    if len(rows)==1:
        print("(no users)")
    else:
        print_table(rows)

def cmd_renew(args):
    # lvctl renew <user> <days>  (alias: extend)
    if len(args) < 3:
        print("usage: lvctl renew <user> <days>"); sys.exit(1)
    user = args[1]
    try:
        days = int(args[2])
    except ValueError:
        print("days harus integer (jumlah hari)"); sys.exit(1)

    db = load_db()
    info = db.get("users", {}).get(user)
    if not info:
        print("no such user"); sys.exit(1)

    now    = epoch()
    curexp = int(info.get("expire_at", 0))
    addsec = days * 86400

    if curexp > now:
        newexp = curexp + addsec
        from_base = "current expiry"
    else:
        newexp = now + addsec
        from_base = "now"

    info["expire_at"]  = newexp
    info["last_update"] = iso_now()
    save_db(db)
    print(f"OK renew {user}: +{days}d from {from_base} -> {fmt_expiry(newexp)}")

def cmd_ips(args):
    if len(args) < 2:
        print("usage: lvctl ips <user>"); sys.exit(1)
    user = args[1]
    db = load_db()
    info = db.get("users", {}).get(user)
    if not info:
        print("no such user"); sys.exit(1)

    last_ip   = info.get("last_ip", "-")
    last_at   = info.get("last_connect_at", "-")
    last_ua   = info.get("last_ua", "-")
    recent    = info.get("recent_ips", [])
    if not isinstance(recent, list): recent = []

    rows = [
        ["Field",            "Value"],
        ["Username",         user],
        ["Last IP",          last_ip],
        ["Last Connect At",  last_at],
        ["Last User-Agent",  last_ua],
        ["Recent IPs",       ", ".join(recent) if recent else "-"],
    ]
    # small table printer
    widths = [max(len(str(c)) for c in col) for col in zip(*rows)]
    for i,row in enumerate(rows):
        print(" ".join(str(c).ljust(w) for c,w in zip(row,widths)))
        if i == 0:
            print(" ".join("-"*w for w in widths))

def cmd_who(args):
    db = load_db()
    users = db.get("users", {})
    rows = [["No.","Username","Last IP","Last Connect At"]]
    n = 0
    for u in sorted(users.keys()):
        inf = users[u]
        n += 1
        rows.append([
            str(n),
            u,
            inf.get("last_ip","-"),
            inf.get("last_connect_at","-"),
        ])
    print_table(rows)

def cmd_resetusage(args):
    # lvctl resetusage <user>|--all [--cum]
    if len(args) < 2:
        print("usage: lvctl resetusage <user>|--all [--cum]"); sys.exit(1)

    target = args[1]
    reset_cum = ("--cum" in args[2:])

    db = load_db()
    users = db.get("users", {})

    def do_reset(u, info):
        info["used_bytes"] = 0
        if reset_cum:
            info["cum_used_bytes"] = 0
            # sekalian refresh window reset
            info["last_reset_at"] = epoch()
            if (info.get("reset_strategy","") == "monthly"):
                info["last_reset_key"] = datetime.datetime.utcnow().strftime("%Y-%m")
        info["last_update"] = iso_now()

    if target == "--all":
        if not users:
            print("(no users)"); return
        for u,info in users.items():
            do_reset(u, info)
        save_db(db)
        print("OK resetusage --all" + (" (with --cum)" if reset_cum else ""))
        return

    if target not in users:
        print("no such user"); sys.exit(1)

    do_reset(target, users[target])
    save_db(db)
    print(f"OK resetusage {target}" + (" (with --cum)" if reset_cum else ""))

def usage():
    print("""lvctl (LingVPN control; proxy-accounting only)
Commands:
  add <user> <password> [days=30] [quota_gb=0] [max_devices=3] [reset=daily|weekly|monthly|none]
  setpass <user> <password>
  setquota <user> <quota_gb>
  enable <user> | disable <user>
  lock <user> <seconds>
  renew <user> <days>    (alias: extend)
  delete <user>
  list
  info <user> [--json] [--show-pass]
  ips <user>
  who
  collect
  enforce
  resetusage <user>|--all [--cum]
""")

if __name__ == "__main__":
    req_root()
    if len(sys.argv) < 2:
        usage(); sys.exit(1)
    cmd = sys.argv[1]; args = sys.argv[1:]
    if   cmd == "add":      cmd_add(args)
    elif cmd == "setpass":  cmd_setpass(args)
    elif cmd == "setquota": cmd_setquota(args)
    elif cmd == "enable":   cmd_enable(args, True)
    elif cmd == "disable":  cmd_enable(args, False)
    elif cmd == "renew"  or cmd == "extend":  cmd_renew(args)
    elif cmd == "lock":     cmd_lock(args)
    elif cmd == "delete":   cmd_delete(args)
    elif cmd == "list":     cmd_list(args)
    elif cmd == "info":     cmd_info(args)
    elif cmd == "collect":  cmd_collect(args)
    elif cmd == "enforce":  cmd_enforce(args)
    elif cmd == "ips":      cmd_ips(args)
    elif cmd == "who":      cmd_who(args)
    elif cmd == "resetusage": cmd_resetusage(args)
    else: usage(); sys.exit(1)
