#!/usr/bin/env python3
# lvmenu - interactive TUI for LingVPN (frontend for lvctl)
# - All operations call /usr/local/sbin/lvctl
# - Choose user from list OR type manually
# - Pretty account summary after add
# - Python 3.9 compatible (no PEP604)
# - Ctrl+C di mana pun membatalkan input & kembali ke menu (tanpa traceback)

import os, sys, json, subprocess, shutil, time, datetime, shlex
from typing import Optional

LVCTL = "/usr/local/sbin/lvctl"
DB_PATH = "/etc/lingvpn/db.json"
DOMAIN_FILE = "/root/domain"

def get_domain() -> str:
    try:
        if os.path.exists(DOMAIN_FILE):
            with open(DOMAIN_FILE, "r", encoding="utf-8") as f:
                for line in f:
                    d = line.strip()
                    if d:
                        return d
    except Exception:
        pass
    return os.environ.get("LV_DOMAIN", "example.id")

# --- Koneksi (untuk banner add) ---
CONF = {
    "DOMAIN": get_domain(),
    "TLS_PORT": int(os.environ.get("LV_TLS_PORT", "443")),
    "NTLS_PORT": int(os.environ.get("LV_NTLS_PORT", "80")),
    "WS_PATH": os.environ.get("LV_WS_PATH", "/sshws"),
    "PAYLOAD_TEMPLATE": 'GET {path}?u={user}&p={pw} HTTP/1.1[crlf]Host: {domain}[crlf]Upgrade: WebSocket[crlf]Connection: Keep-Alive[crlf]User-Agent: [ua][crlf][crlf]',
}

# ---------- helpers ----------
def req_root():
    if os.geteuid() != 0:
        print("Run as root"); sys.exit(1)

def run_cmd_show(cmd: str) -> int:
    print(f"$ {cmd}")
    p = subprocess.run(cmd, shell=True, text=True)
    return p.returncode

def run_lvctl(args):
    return subprocess.run([LVCTL] + args, text=True, capture_output=True)

def load_db():
    if not os.path.exists(DB_PATH):
        return {"users":{}}
    try:
        with open(DB_PATH, "r") as f:
            return json.load(f)
    except Exception:
        return {"users":{}}

def fmt_bytes_auto(b):
    try: b = int(b)
    except: b = 0
    units = ["B","KB","MB","GB","TB","PB"]
    val = float(b); i=0
    while val >= 1024.0 and i < len(units)-1:
        val /= 1024.0; i += 1
    return f"{int(val)} {units[i]}" if i==0 else f"{val:.2f} {units[i]}"

def fmt_exp(exp_epoch):
    try: exp = int(exp_epoch)
    except: exp = 0
    if exp <= 0: return "Never"
    dt = datetime.datetime.utcfromtimestamp(exp)
    days = max(0, (dt - datetime.datetime.utcnow()).days)
    return f"{dt.strftime('%Y-%m-%d %H:%M:%SZ')} ({days}d)"

def status_of(info):
    now = int(time.time())
    if not info.get("enabled", True): return "Disabled"
    if int(info.get("lock_until",0)) > now: return "Locked"
    exp = int(info.get("expire_at",0))
    if exp>0 and now>exp: return "Expired"
    q = int(info.get("quota_bytes",0))
    used = int(info.get("used_bytes",0))
    if q>0 and used>=q: return "Over-quota"
    return "Active"

def safe_input(prompt: str) -> Optional[str]:
    """Input yang aman dari Ctrl+C/EOF. Return None jika dibatalkan."""
    try:
        return input(prompt)
    except (KeyboardInterrupt, EOFError):
        print("\n(batal)")
        return None

def pause():
    _ = safe_input("\n[Enter] untuk lanjut...")

# ---------- pickers ----------
def list_users_and_pick(allow_manual=True) -> Optional[str]:
    db = load_db()
    users = sorted(list(db.get("users", {}).keys()))
    if not users:
        print("(tidak ada user)")
        if allow_manual:
            u = safe_input("Ketik username (atau kosong untuk batal): ")
            return (u or "").strip() or None
        return None

    print("\n=== USERS ===")
    for i,u in enumerate(users, start=1):
        info = db["users"][u]
        st = status_of(info)
        used = fmt_bytes_auto(info.get("used_bytes",0))
        lim  = "Unlimited" if int(info.get("quota_bytes",0))==0 else fmt_bytes_auto(info.get("quota_bytes",0))
        print(f"{i:2d}. {u:20s}  {st:10s}  used:{used:>10s}  limit:{lim:>10s}")
    if allow_manual:
        print(" M. ketik username manual")
    print(" X. batal")

    while True:
        ch = safe_input("Pilih nomor/M/X: ")
        if ch is None or not ch.strip():
            return None
        ch = ch.strip()
        if ch.lower() == "x":
            return None
        if allow_manual and ch.lower() == "m":
            u = safe_input("Username: ")
            return (u or "").strip() or None
        if ch.isdigit():
            idx = int(ch)
            if 1 <= idx <= len(users):
                return users[idx-1]
        print("Pilihan tidak valid.")

# ---------- detail akun setelah add ----------
def show_account_details(user, pw, conf):
    import pathlib, hmac, hashlib

    db   = load_db()
    info = db.get("users", {}).get(user, {})

    qbytes    = int(info.get("quota_bytes", 0))
    limit_h   = "Unlimited" if qbytes == 0 else fmt_bytes_auto(qbytes)
    expires_h = fmt_exp(info.get("expire_at", 0))
    status_h  = status_of(info)
    reset_h   = info.get("reset_strategy", "-")
    maxd_h    = int(info.get("max_devices", 0))

    payload = conf["PAYLOAD_TEMPLATE"].format(
    path   = conf["WS_PATH"],
    user   = user,
    pw     = pw,
    domain = conf["DOMAIN"],
    )

    sub_url = None
    try:
        secret_path = pathlib.Path("/etc/lingvpn/sub.secret")
        if secret_path.exists() and secret_path.stat().st_size > 0:
            key   = secret_path.read_bytes().strip()
            token = hmac.new(key, user.encode(), hashlib.sha256).hexdigest()  # full 64-hex
            domain = conf.get("DOMAIN") or (pathlib.Path("/root/domain").read_text().strip()
                                             if pathlib.Path("/root/domain").exists() else "localhost")
            sub_url = f"https://{domain}/lvsub/?u={user}&t={token}"
    except Exception:
        sub_url = None

    # output rapi
    print("\nPembuatan akun BERHASIL!")
    print("-" * 33)
    print("+++++ SSH-WS Account Created +++++")
    print(f"Domain         : {conf['DOMAIN']}")
    print(f"Username       : {user}")
    print(f"Password       : {pw}")
    print(f"Port           : {conf['TLS_PORT']} [TLS], {conf['NTLS_PORT']} [nTLS]")
    print(f"Status         : {status_h}")
    print(f"Max Devices    : {maxd_h}")
    print(f"Reset Strategy : {reset_h}")
    print(f"Data Limit     : {limit_h}")
    print(f"Expires At     : {expires_h}")
    if sub_url:
        print(f"Status Page    : {sub_url}")
    print("-" * 31)
    print("Payload default (injector-style):")
    print(payload)

# ---------- actions ----------
def do_add():
    user = safe_input("Username: ")
    if not user: print("batal."); return
    user = user.strip()
    if not user: print("batal."); return

    pw = safe_input("Password: ")
    if pw is None: print("batal."); return
    pw = pw.strip()

    days = safe_input("Masa aktif (hari) [30]: ")
    if days is None: print("batal."); return
    days = (days.strip() or "30")

    quota = safe_input("Quota (GB) [0=unlimited]: ")
    if quota is None: print("batal."); return
    quota = (quota.strip() or "0")

    maxd = safe_input("Max devices [3]: ")
    if maxd is None: print("batal."); return
    maxd = (maxd.strip() or "3")

    reset = safe_input("Reset strategy [daily/weekly/monthly/none] [monthly]: ")
    if reset is None: print("batal."); return
    reset = (reset.strip() or "monthly")

    res = run_lvctl(["add", user, pw, days, quota, maxd, reset])
    if res.returncode != 0:
        print(res.stderr or res.stdout); return
    print(res.stdout.strip())
    show_account_details(user, pw, CONF)

def do_setpass():
    user = list_users_and_pick(allow_manual=True)
    if not user:
        print("batal."); return
    pw = safe_input("Password baru: ")
    if not pw:
        print("batal."); return
    pw = pw.strip()
    if not pw:
        print("batal."); return
    res = run_lvctl(["setpass", user, pw])
    print(res.stderr or res.stdout)

def do_setquota():
    user = list_users_and_pick(allow_manual=True)
    if not user:
        print("batal."); return
    q = safe_input("Quota GB (0=unlimited): ")
    if not q:
        print("batal."); return
    q = q.strip()
    if not q:
        print("batal."); return
    res = run_lvctl(["setquota", user, q])
    print(res.stderr or res.stdout)

def do_enable_disable(enable=True):
    user = list_users_and_pick(allow_manual=True)
    if not user:
        print("batal."); return
    cmd = "enable" if enable else "disable"
    res = run_lvctl([cmd, user])
    print(res.stderr or res.stdout)

def do_lock():
    user = list_users_and_pick(allow_manual=True)
    if not user:
        print("batal."); return
    print("Durasi lock (pilih):")
    print(" 1) 5 menit")
    print(" 2) 30 menit")
    print(" 3) 1 jam")
    print(" 4) Custom (detik)")
    ch = safe_input("Pilih [1-4]: ")
    if not ch: print("batal."); return
    ch = ch.strip()
    if ch == "1": secs = 5*60
    elif ch == "2": secs = 30*60
    elif ch == "3": secs = 60*60
    elif ch == "4":
        raw = safe_input("Masukkan detik: ")
        if not raw or not raw.strip().isdigit():
            print("invalid."); return
        secs = int(raw.strip())
    else:
        print("batal."); return
    res = run_lvctl(["lock", user, str(secs)])
    print(res.stderr or res.stdout)

def do_renew():
    user = list_users_and_pick(allow_manual=True)
    if not user:
        print("batal."); return
    days = safe_input("Tambah hari: ")
    if not days or not days.strip().isdigit():
        print("harus angka."); return
    res = run_lvctl(["renew", user, days.strip()])
    print(res.stderr or res.stdout)

def do_delete():
    user = list_users_and_pick(allow_manual=True)
    if not user:
        print("batal."); return
    y = safe_input(f"Yakin hapus '{user}'? [y/N]: ")
    if not y or y.strip().lower() != "y":
        print("batal."); return
    res = run_lvctl(["delete", user])
    print(res.stderr or res.stdout)

def do_list():
    res = run_lvctl(["list"])
    print(res.stdout if res.stdout else res.stderr)

def do_info():
    user = list_users_and_pick(allow_manual=True)
    if not user:
        print("batal."); return
    showpw = safe_input("Tampilkan password? [y/N]: ")
    showpw = (showpw or "").strip().lower() == "y"
    args = ["info", user]
    if showpw: args.append("--show-pass")
    res = run_lvctl(args)
    print(res.stdout if res.stdout else res.stderr)

def do_ips():
    user = list_users_and_pick(allow_manual=True)
    if not user:
        print("batal."); return
    res = run_lvctl(["ips", user])
    print(res.stdout if res.stdout else res.stderr)

def do_who():
    res = run_lvctl(["who"])
    print(res.stdout if res.stdout else res.stderr)

def do_collect():
    res = run_lvctl(["collect"])
    print(res.stdout if res.stdout else res.stderr)

def do_enforce():
    res = run_lvctl(["enforce"])
    print(res.stdout if res.stdout else res.stderr)

def do_resetusage():
    os.system("clear")
    print("Reset Usage")
    print("1) Reset satu user")
    print("2) Reset SEMUA user")
    print("0) Batal")
    ch = safe_input("Pilih: ")
    if ch is None or ch.strip() == "0":
        return

    if ch.strip() == "1":
        u = list_users_and_pick(allow_manual=True)
        if not u:
            print("dibatalkan."); return
        rc = run_cmd_show(f"{shlex.quote(LVCTL)} resetusage {shlex.quote(u)}")
        print("OK" if rc == 0 else f"Gagal (rc={rc})")
    elif ch.strip() == "2":
        yakin = (safe_input("Yakin reset semua user? ketik 'YA' untuk lanjut: ") or "").strip().upper()
        if yakin != "YA":
            print("dibatalkan."); return
        rc = run_cmd_show(f"{shlex.quote(LVCTL)} resetusage --all")
        print("OK" if rc == 0 else f"Gagal (rc={rc})")
    else:
        print("pilihan tidak dikenal.")

def do_backup_restore():
    rc = run_cmd_show("lvbackup-menu")
    if rc != 0:
        print("Gagal menjalankan lvbackup-menu (pastikan terpasang & executable).")

# ---------- menu ----------
import re, shutil, textwrap, sys, time, os

_ANSI_RE = re.compile(r"\x1b\[[0-9;]*m")

def _ansi(name):
    if not sys.stdout.isatty():
        return ""
    return {
        "reset":"\033[0m","muted":"\033[38;5;245m",
        "title":"\033[38;5;111m","num":"\033[38;5;81m","accent":"\033[38;5;45m",
    }.get(name,"")

def _vislen(s: str) -> int:           # panjang yang terlihat (tanpa ANSI)
    return len(_ANSI_RE.sub("", s or ""))

def _pad_to(s: str, width: int) -> str:  # pad s sampai width kolom “terlihat”
    pad = max(0, width - _vislen(s))
    return s + (" " * pad)

def _hr(w): return "─" * max(0, w)

def _menu_line(num, label, width_num=3, width_label=40):
    n = f"{_ansi('num')}{str(num).rjust(width_num)}{_ansi('reset')})"
    parts = textwrap.wrap(label, width_label) or [label]
    out = [f"{n}  {parts[0]}"]
    for line in parts[1:]:
        out.append(" " * (width_num + 3) + line)
    return out  # -> list of lines

# ---------- menu ----------
def main_menu():
    items = [
        ("1","Tambah user",                      lambda:(os.system("clear"),do_add(),pause())),
        ("2","Set password",                     lambda:(os.system("clear"),do_setpass(),pause())),
        ("3","Set quota",                        lambda:(os.system("clear"),do_setquota(),pause())),
        ("4","Enable user",                      lambda:(os.system("clear"),do_enable_disable(True),pause())),
        ("5","Disable user",                     lambda:(os.system("clear"),do_enable_disable(False),pause())),
        ("6","Lock user",                        lambda:(os.system("clear"),do_lock(),pause())),
        ("7","Renew/Extend user",                lambda:(os.system("clear"),do_renew(),pause())),
        ("8","Delete user",                      lambda:(os.system("clear"),do_delete(),pause())),
        ("9","List users",                       lambda:(os.system("clear"),do_list(),pause())),
        ("10","Info user",                       lambda:(os.system("clear"),do_info(),pause())),
        ("11","Lihat IP user",                   lambda:(os.system("clear"),do_ips(),pause())),
        ("12","Who (ringkas)",                   lambda:(os.system("clear"),do_who(),pause())),
        ("13","Collect (sanitize DB)",           lambda:(os.system("clear"),do_collect(),pause())),
        ("14","Enforce (kick yg melanggar)",     lambda:(os.system("clear"),do_enforce(),pause())),
        ("15","Reset usage",		         lambda:(os.system("clear"),do_resetusage(),pause())),
        ("16","Backup/Restore Telegram",         lambda:(os.system("clear"),do_backup_restore(),pause())),
        ("0","Keluar",                           None),
    ]
    while True:
        try:
            os.system("clear")
            cols = shutil.get_terminal_size(fallback=(100,28)).columns
            panel_w = min(86, max(60, cols - 4))              # lebar dalam
            topbot  = "┌" + _hr(panel_w) + "┐"
            sep     = "├" + _hr(panel_w) + "┤"
            bott    = "└" + _hr(panel_w) + "┘"

            print(topbot)
            title = f"{_ansi('title')}LingVPN Menu{_ansi('reset')}"
            print("│ " + _pad_to(title, panel_w - 1) + "│")
            print(sep)

            width_num = 2 if max(len(k) for k,_,_ in items) <= 2 else 3
            width_label = panel_w - (width_num + 5)

            for k, label, _ in items:
                lines = _menu_line(k, label, width_num, width_label)
                # cetak setiap line dengan padding visible
                for i, ln in enumerate(lines):
                    print("│ " + _pad_to(ln, panel_w - 1) + "│")

            print(sep)
            tip = f"{_ansi('muted')}Ctrl+C: batal/ulang · Enter kosong: ulang menu{_ansi('reset')}"
            print("│ " + _pad_to(tip, panel_w - 1) + "│")
            print(bott)

            ch = safe_input(f"{_ansi('accent')}Pilih{_ansi('reset')}: ")
            if ch is None:  # Ctrl+C di prompt
                continue
            ch = ch.strip()
            matched = False
            for k, _, fn in items:
                if ch == k:
                    matched = True
                    if k == "0":
                        print("bye."); return
                    fn()
                    break
            if not matched:
                print("Pilihan tidak dikenal."); pause()
        except KeyboardInterrupt:
            print("\n(dibatalkan)"); time.sleep(0.8); continue

if __name__ == "__main__":
    req_root()
    if not shutil.which(LVCTL):
        print(f"Error: {LVCTL} tidak ditemukan."); sys.exit(1)
    CONF["DOMAIN"] = get_domain()
    main_menu()
