#!/usr/bin/env python3
# lvsubd.py - mini API subscription LingVPN (read-only)
import http.server, socketserver, urllib.parse, json, time, os, hmac, hashlib

DB_PATH     = "/etc/lingvpn/db.json"
SECRET_PATH = "/etc/lingvpn/sub.secret"
BIND        = "127.0.0.1"
PORT        = 11020

def load_db():
    try:
        with open(DB_PATH, "r") as f:
            return json.load(f)
    except Exception:
        return {"users":{}}

def load_secret():
    with open(SECRET_PATH, "rb") as f:
        return f.read().strip()

def hmac_token(secret: bytes, user: str) -> str:
    import hashlib, hmac
    return hmac.new(secret, user.encode(), hashlib.sha256).hexdigest()

def fmt_bytes(b: int) -> str:
    try: b = int(b)
    except: b = 0
    u = ["B","KB","MB","GB","TB","PB"]
    v = float(b); i = 0
    while v >= 1024 and i < len(u)-1:
        v /= 1024; i += 1
    return (f"{v:.2f} {u[i]}" if i else f"{int(v)} {u[i]}")

def fmt_exp(exp: int) -> str:
    try: exp = int(exp)
    except: exp = 0
    if exp <= 0: return "Never", None
    dt = time.gmtime(exp)
    left = max(0, exp - int(time.time()))
    days = left // 86400
    return time.strftime("%Y-%m-%d %H:%M:%SZ", dt) + f" ({days}d)", days

def status_of(info: dict) -> str:
    now = int(time.time())
    if not info.get("enabled", True): return "Disabled"
    if int(info.get("lock_until",0)) > now: return "Locked"
    exp = int(info.get("expire_at",0))
    if exp>0 and now>exp: return "Expired"
    q = int(info.get("quota_bytes",0))
    used = int(info.get("used_bytes",0))
    if q>0 and used>=q: return "Over-quota"
    return "Active"

class Handler(http.server.BaseHTTPRequestHandler):
    def do_GET(self):
        # endpoint tunggal: /
        qs = urllib.parse.urlparse(self.path).query
        q  = urllib.parse.parse_qs(qs)
        user = (q.get("u") or [""])[0].strip()
        tok  = (q.get("t") or [""])[0].strip()
        if not user or not tok:
            return self._json(400, {"error":"missing u/t"})

        try:
            secret = load_secret()
        except FileNotFoundError:
            return self._json(500, {"error":"secret missing"})

        if tok.lower() != hmac_token(secret, user):
            return self._json(403, {"error":"bad token"})

        db = load_db()
        info = db.get("users",{}).get(user)
        if not info:
            return self._json(404, {"error":"user not found"})

        used    = int(info.get("used_bytes",0))
        cum     = int(info.get("cum_used_bytes",0))
        limit_b = int(info.get("quota_bytes",0))
        limit_h = "Unlimited" if limit_b==0 else fmt_bytes(limit_b)
        exp_h, days_left = fmt_exp(int(info.get("expire_at",0)))
        reset_str = info.get("reset_strategy","-")
        obj = {
            "username": user,
            "status":   status_of(info),
            "used_bytes": used, "used_h": fmt_bytes(used),
            "cum_bytes":  cum,  "cum_h":  fmt_bytes(cum),
            "limit_bytes": limit_b, "limit_h": limit_h,
            "reset_strategy": (reset_str.capitalize() if reset_str!="none" else "-"),
            "expires_at": exp_h, "days_left": (days_left if days_left is not None else -1),
            "last_ip": info.get("last_ip","-"),
            "last_connect_at": info.get("last_connect_at","-"),
            "max_devices": int(info.get("max_devices",0)),
            "server_time": time.strftime("%Y-%m-%dT%H:%M:%SZ", time.gmtime())
        }
        return self._json(200, obj)

    def log_message(self, *a, **k): return  # silent

    def _json(self, code:int, obj:dict):
        b = json.dumps(obj).encode()
        self.send_response(code)
        self.send_header("Content-Type","application/json")
        self.send_header("Content-Length", str(len(b)))
        self.end_headers()
        self.wfile.write(b)

def main():
    os.makedirs(os.path.dirname(DB_PATH), exist_ok=True)
    with socketserver.TCPServer((BIND, PORT), Handler) as httpd:
        httpd.allow_reuse_address = True
        httpd.serve_forever()

if __name__ == "__main__":
    main()
