#!/usr/bin/env python3
import socket, threading, select, sys, os, json, time, argparse
from urllib.parse import urlparse, parse_qs

DB_PATH   = "/etc/lingvpn/db.json"
WS_PATH   = "/sshws"          # path yang diterima
DEFAULT_HOST = "10.200.1.2"
DEFAULT_PORT = 143
BUFLEN    = 65536
TIMEOUT   = 60
SAVE_EVERY = 5                # detik: flush progress ke DB
CLEAR_IPS_AFTER = int(os.environ.get("LV_CLEAR_IPS_AFTER", "180"))  # detik (default 10 menit)

RESP_101 = (
    "HTTP/1.1 101 Switching Protocols\r\n"
    "Upgrade: websocket\r\n"
    "Connection: Upgrade\r\n\r\n"
).encode()
RESP_400 = b"HTTP/1.1 400 Bad Request\r\nContent-Length:0\r\n\r\n"
RESP_401 = b"HTTP/1.1 401 Unauthorized\r\nContent-Length:0\r\n\r\n"
RESP_403 = b"HTTP/1.1 403 Forbidden\r\nContent-Length:0\r\n\r\n"
RESP_429 = b"HTTP/1.1 429 Too Many Requests\r\nContent-Length:0\r\n\r\n"
RESP_409 = b"HTTP/1.1 409 Conflict\r\nContent-Length:0\r\n\r\n"
RESP_503 = b"HTTP/1.1 503 Service Unavailable\r\nContent-Length:0\r\n\r\n"

_db_lock = threading.Lock()
_live_lock = threading.Lock()
_live_conns = {}   # {user: count}

# tracking koneksi aktif per-user + kapan terakhir semua koneksi user itu terputus
_active_lock = threading.Lock()
_active_sessions = {}   # {user: {conn_id: {"sockets": (cli, srv)}}}
_last_disconnect = {}   # {user: epoch}

def now(): return int(time.time())
def iso_now(): return time.strftime("%Y-%m-%dT%H:%M:%SZ", time.gmtime())

def load_db():
    with _db_lock:
        if not os.path.exists(DB_PATH):
            return {"users":{}, "netns":"dbns0"}
        with open(DB_PATH,"r") as f:
            return json.load(f)

def save_db(db):
    tmp = DB_PATH + ".tmp"
    with _db_lock:
        with open(tmp,"w") as f:
            json.dump(db,f,indent=2,sort_keys=True)
        os.replace(tmp, DB_PATH)

def parse_headers(buf: bytes):
    try:
        header, _ = buf.split(b"\r\n\r\n", 1)
    except ValueError:
        return None, {}
    lines = header.split(b"\r\n")
    try:
        method, path, proto = lines[0].decode(errors="ignore").split(" ",2)
    except:
        return None, {}
    hdrs={}
    for ln in lines[1:]:
        if b":" in ln:
            k,v = ln.split(b":",1)
            hdrs[k.strip().lower()] = v.strip()
    return (method, path, proto), hdrs

def get_hdr(hdrs, name_bytes):
    v = hdrs.get(name_bytes)
    return v.decode() if v else ""

def parse_hostport(hdrs):
    # Bisa override via X-Real-Host: ip:port, kalau kosong pakai default
    hostport = get_hdr(hdrs, b"x-real-host") or f"{DEFAULT_HOST}:{DEFAULT_PORT}"
    if ":" in hostport:
        h,p = hostport.rsplit(":",1)
        try: return h, int(p)
        except: return h, DEFAULT_PORT
    return hostport, DEFAULT_PORT

def monthly_key():
    return time.strftime("%Y-%m", time.gmtime())

def quota_left(uinfo):
    q = int(uinfo.get("quota_bytes",0))
    used = int(uinfo.get("used_bytes",0))
    return (q - used) if q>0 else 1<<60

def allowed_user(uinfo):
    t = now()
    if not uinfo.get("enabled", True): return False
    if int(uinfo.get("lock_until",0)) > t: return False
    exp = int(uinfo.get("expire_at",0))
    if exp>0 and t>exp: return False
    # reset window
    strat = uinfo.get("reset_strategy","none")
    if strat=="monthly":
        key = uinfo.get("last_reset_key","")
        if key != monthly_key():
            uinfo["used_bytes"]=0
            uinfo["last_reset_key"]=monthly_key()
            uinfo["last_reset_at"]=t
    else:
        sec = int(uinfo.get("reset_every_seconds",0))
        if sec>0 and (t - int(uinfo.get("last_reset_at",0)) >= sec):
            uinfo["used_bytes"]=0
            uinfo["last_reset_at"]=t
    return True

def inc_live(user):
    with _live_lock:
        _live_conns[user] = _live_conns.get(user,0)+1
        return _live_conns[user]

def dec_live(user):
    with _live_lock:
        if user in _live_conns:
            _live_conns[user] = max(0, _live_conns[user]-1)
            if _live_conns[user]==0: _live_conns.pop(user,None)

def best_client_ip(hdrs, addr_tuple):
    """
    Urutan prioritas:
      1) X-Real-IP
      2) X-Forwarded-For → IP pertama
      3) addr_tuple[0] dari socket accept
    """
    xri = get_hdr(hdrs, b"x-real-ip").strip()
    if xri:
        return xri
    xff = get_hdr(hdrs, b"x-forwarded-for").strip()
    if xff:
        return xff.split(",")[0].strip()
    return addr_tuple[0] if addr_tuple and len(addr_tuple)>=1 else ""

def set_tcp_keepalive(sock: socket.socket):
    try:
        sock.setsockopt(socket.SOL_SOCKET, socket.SO_KEEPALIVE, 1)
        # nilai default OS biasanya cukup; kalau mau lebih agresif, aktifkan di bawah:
        # sock.setsockopt(socket.IPPROTO_TCP, socket.TCP_KEEPIDLE, 30)
        # sock.setsockopt(socket.IPPROTO_TCP, socket.TCP_KEEPINTVL, 10)
        # sock.setsockopt(socket.IPPROTO_TCP, socket.TCP_KEEPCNT, 3)
    except Exception:
        pass

def pipe(src, dst, meter):
    try:
        while True:
            r, _, _ = select.select([src], [], [], TIMEOUT)
            if not r: break
            data = src.recv(BUFLEN)
            if not data: break
            dst.sendall(data)
            meter(len(data))
    except:
        pass

def handle_client(cli, addr):
    user=None
    conn_id=None
    srv=None
    try:
        set_tcp_keepalive(cli)

        buf = cli.recv(65536)
        if not buf: cli.close(); return

        req,hdrs = parse_headers(buf)
        if not req: cli.sendall(RESP_400); cli.close(); return
        method, path, _ = req
        if method!="GET": cli.sendall(RESP_400); cli.close(); return

        # path harus WS_PATH (query boleh, tapi path prefix harus sama)
        u = urlparse(path)
        if not u.path.startswith(WS_PATH):
            cli.sendall(RESP_400); cli.close(); return

        # Upgrade minimal
        if b"upgrade: websocket" not in buf.lower():
            cli.sendall(RESP_400); cli.close(); return

        # Ambil kredensial dari header atau query (?u=&p=)
        user = get_hdr(hdrs,b"x-user")
        passwd = get_hdr(hdrs,b"x-pass")
        if (not user or not passwd) and u.query:
            q = parse_qs(u.query)
            user   = user   or (q.get("u",[None])[0] or "")
            passwd = passwd or (q.get("p",[None])[0] or "")

        # Cek DB user
        db = load_db()
        uinfo = db.get("users",{}).get(user)
        if not uinfo: cli.sendall(RESP_401); cli.close(); return
        if passwd != str(uinfo.get("password","")):
            cli.sendall(RESP_401); cli.close(); return

        # Enforce enable/lock/expire/reset-window
        if not allowed_user(uinfo):
            cli.sendall(RESP_403); cli.close(); return

        # Max devices (concurrent)
        cur = inc_live(user)
        try:
            maxd = int(uinfo.get("max_devices",0) or 0)
        except: maxd = 0
        if maxd>0 and cur>maxd:
            cli.sendall(RESP_429); cli.close(); dec_live(user); return

        # --- Capture client IP & UA, simpan ke DB ---
        client_ip = best_client_ip(hdrs, addr)
        ua = get_hdr(hdrs, b"user-agent")
        uinfo["last_ip"] = client_ip
        uinfo["last_connect_at"] = iso_now()
        if ua:
            uinfo["last_ua"] = ua
        arr = list(dict.fromkeys([client_ip] + list(uinfo.get("recent_ips", []))))
        uinfo["recent_ips"] = arr[:5]
        db["users"][user] = uinfo
        save_db(db)

        # --- daftar sesi aktif (untuk auto-clear IP) ---
        conn_id = f"{time.time()}-{id(cli)}"
        with _active_lock:
            _active_sessions.setdefault(user, {})[conn_id] = {"sockets": (cli, None)}
            # user baru aktif → hapus last_disconnect mark
            _last_disconnect.pop(user, None)

        # Tujuan
        host, port = parse_hostport(hdrs)
        try:
            srv = socket.create_connection((host, port), timeout=10)
            set_tcp_keepalive(srv)
        except:
            cli.sendall(RESP_503); cli.close()
            with _active_lock:
                sess = _active_sessions.get(user,{})
                sess.pop(conn_id, None)
                if not sess:
                    _last_disconnect[user] = now()
            dec_live(user); return

        # update socket pair di sesi
        with _active_lock:
            if user in _active_sessions and conn_id in _active_sessions[user]:
                _active_sessions[user][conn_id]["sockets"] = (cli, srv)

        # 101
        cli.sendall(RESP_101)

        # Akuntansi: update used_bytes berkala & saat close
        last_save = time.time()
        used_delta = 0
        over_quota = [False]
        lock = threading.Lock()

        def meter(n):
            nonlocal used_delta, last_save
            with lock:
                used_delta += n
                t = time.time()
                if t - last_save >= SAVE_EVERY:
                    db = load_db()
                    ui = db.get("users",{}).get(user)
                    if ui:
                        ui["used_bytes"] = int(ui.get("used_bytes",0)) + used_delta
                        ui["cum_used_bytes"] = int(ui.get("cum_used_bytes",0)) + used_delta
                        ui["last_update"] = iso_now()
                        db["users"][user] = ui
                        save_db(db)
                        q = int(ui.get("quota_bytes",0))
                        if q>0 and ui["used_bytes"] >= q:
                            over_quota[0]=True
                    used_delta = 0
                    last_save = t

        # Relay dua arah
        def c2s(): pipe(cli, srv, meter)
        def s2c(): pipe(srv, cli, meter)

        t1=threading.Thread(target=c2s,daemon=True)
        t2=threading.Thread(target=s2c,daemon=True)
        t1.start(); t2.start()

        while t1.is_alive() or t2.is_alive():
            if over_quota[0]:
                try: cli.shutdown(socket.SHUT_RDWR)
                except: pass
                try: srv.shutdown(socket.SHUT_RDWR)
                except: pass
                break
            time.sleep(0.5)
        t1.join(); t2.join()

        # flush sisa delta
        if used_delta>0:
            db = load_db()
            ui = db.get("users",{}).get(user)
            if ui:
                ui["used_bytes"] = int(ui.get("used_bytes",0)) + used_delta
                ui["cum_used_bytes"] = int(ui.get("cum_used_bytes",0)) + used_delta
                ui["last_update"] = iso_now()
                db["users"][user] = ui
                save_db(db)

    finally:
        # tutup socket
        for s in (srv,):
            if s:
                try: s.close()
                except: pass
        if user: 
            dec_live(user)
            with _active_lock:
                sess = _active_sessions.get(user,{})
                if conn_id and conn_id in sess:
                    sess.pop(conn_id, None)
                if not sess:
                    # tidak ada koneksi aktif lagi → tandai waktu disconnect
                    _last_disconnect[user] = now()

def cleaner():
    # worker untuk bersihkan last_ip / recent_ips setelah idle
    while True:
        time.sleep(30)
        now_ts = now()
        to_clear=[]
        with _active_lock:
            for u, tdisc in list(_last_disconnect.items()):
                active = len(_active_sessions.get(u, {})) > 0
                if not active and (now_ts - tdisc) >= CLEAR_IPS_AFTER:
                    to_clear.append(u)
        if not to_clear:
            continue
        db = load_db()
        changed=False
        for u in to_clear:
            # double-check masih idle
            with _active_lock:
                if len(_active_sessions.get(u, {})) > 0:
                    continue
            ui = db.get("users",{}).get(u)
            if not ui: 
                with _active_lock: _last_disconnect.pop(u, None)
                continue
            ui["last_ip"] = ""
            ui["recent_ips"] = []
            ui["last_update"] = iso_now()
            changed=True
            with _active_lock:
                _last_disconnect.pop(u, None)
        if changed:
            save_db(db)

def main():
    ap = argparse.ArgumentParser()
    ap.add_argument("-b","--bind", default="0.0.0.0")
    ap.add_argument("-p","--port", type=int, required=True)
    args = ap.parse_args()

    s=socket.socket(socket.AF_INET, socket.SOCK_STREAM)
    s.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR,1)
    s.bind((args.bind, args.port))
    s.listen(256)
    print(f"[wsproxy] listening {args.bind}:{args.port} path={WS_PATH} clear_ips_after={CLEAR_IPS_AFTER}s")

    # start cleaner thread
    threading.Thread(target=cleaner, daemon=True).start()

    try:
        while True:
            c,a = s.accept()
            threading.Thread(target=handle_client, args=(c,a), daemon=True).start()
    except KeyboardInterrupt:
        pass
    finally:
        s.close()

if __name__=="__main__":
    main()
