#!/usr/bin/env python3
# subsrv.py — Subscription mini server:
# - HTML & JSON status (quota, expired, IP aktif)
# - Proteksi HMAC token (?token=...)
# - Menampilkan template Clash (TXT terbaru) & semua JSON v2rayNG + tombol Copy/Download

import os
import hmac
import hashlib
import json
import time
import re
import glob
from datetime import datetime, timezone, timedelta
from flask import Flask, request, jsonify, abort, render_template_string

# ========= KONFIG dari ENV =========
DB_PATH      = os.environ.get("XRAY_DB", "/data/database.json")
ACCESS_LOG   = os.environ.get("XRAY_ACCESS_LOG", "/logs/access.log")
WINDOW_SEC   = int(os.environ.get("SUB_WINDOW_SEC", "600"))  # default 10 menit
SECRET_KEY   = os.environ.get("SUB_SECRET", "GANTI_INI_PANJANG_BANGET_32-64CHAR")
TZ           = timezone(timedelta(hours=int(os.environ.get("TZ_OFFSET_HOURS", "7"))))  # WIB default
BASE_HTML    = os.environ.get("SUB_BASE_HTML", "s22")   # /s22/<email>
BASE_JSON    = os.environ.get("SUB_BASE_JSON", "s22j")  # /s22j/<email>
CLIENT_DIR   = os.environ.get("CLIENT_DIR", "/var/www/html")

app = Flask(__name__)

# ========= Utils =========
def hmac_token(email: str) -> str:
    return hmac.new(SECRET_KEY.encode(), email.encode(), hashlib.sha256).hexdigest()

def verify(email: str, token: str) -> bool:
    return hmac.compare_digest(hmac_token(email), token or "")

def load_db():
    with open(DB_PATH, "r") as f:
        return json.load(f)

def fmt_bytes(n: int) -> str:
    units = ["B", "KB", "MB", "GB", "TB", "PB"]
    x = float(n); i = 0
    while x >= 1024 and i < len(units)-1:
        x /= 1024; i += 1
    return f"{x:.2f} {units[i]}"

# IP privat/lokal
_ipv4_priv = re.compile(r"^(127\.|10\.|192\.168\.|172\.(1[6-9]|2[0-9]|3[0-1])\.)")
def is_private_ip(ip: str) -> bool:
    ip_l = ip.lower()
    if ip in ("0.0.0.0", "localhost", "::1"):
        return True
    if _ipv4_priv.match(ip):
        return True
    # IPv6 heuristik
    if ip_l.startswith("fc") or ip_l.startswith("fd"):  # fc00::/7
        return True
    if ip_l.startswith("fe8"):                          # fe80::/10
        return True
    return False

# Parsing access.log Xray:
# Contoh:
# 2025/09/29 12:48:16.850496 from 36.81.183.133:0 accepted 1.1.1.1:853 [SSWS >> direct] email: linggar
# 2025/09/29 06:00:03.367871 from 112.215.220.20:0 accepted tcp:bing.com:80 [VLESS_WS >> direct] email: 276.Erlansaja
_ts_re   = re.compile(r"^(\d{4}/\d{2}/\d{2})\s+(\d{2}):(\d{2}):(\d{2})")
_from_re = re.compile(r"\sfrom\s([^\s]+)")

def extract_ips_from_log(email: str, window_sec: int):
    """
    Return dict ip -> last_seen_epoch (UTC) dalam window detik terakhir.
    Buang IP private & loopback. Bersihkan "tcp:" prefix dan port.
    """
    out = {}
    if not os.path.isfile(ACCESS_LOG):
        return out

    cutoff_utc = datetime.now(timezone.utc) - timedelta(seconds=window_sec)
    tag = f"email: {email}"

    with open(ACCESS_LOG, "r", errors="ignore") as f:
        for line in f:
            if tag not in line:
                continue

            m = _ts_re.match(line)
            if not m:
                continue
            try:
                ts_local = datetime.strptime(
                    f"{m.group(1)} {m.group(2)}:{m.group(3)}:{m.group(4)}",
                    "%Y/%m/%d %H:%M:%S"
                )
            except Exception:
                continue

            ts_utc = ts_local.replace(tzinfo=TZ).astimezone(timezone.utc)
            if ts_utc < cutoff_utc:
                continue

            mf = _from_re.search(line)
            if not mf:
                continue
            host = mf.group(1)  # contoh "36.81.183.133:0" / "[2001:db8::1]:1234" / "tcp:1.2.3.4:0"

            # buang proto prefix
            if host.startswith("tcp:") or host.startswith("udp:"):
                host = host.split(":", 1)[1]  # remove first "tcp:"/"udp:"

            # IPv6 bracket
            if host.startswith("["):
                # [ipv6]:port
                end = host.find("]")
                if end != -1:
                    ip = host[1:end]
                else:
                    ip = host
            else:
                # strip :port (ambil sebelum ":" terakhir)
                if ":" in host:
                    ip = host.rsplit(":", 1)[0]
                else:
                    ip = host

            ip = ip.rstrip(",;")
            if is_private_ip(ip):
                continue

            out[ip] = max(out.get(ip, 0), int(ts_utc.timestamp()))
    return out

def user_view(email: str):
    db = load_db()
    user = db.get("users", {}).get(email)
    if not user:
        return None
    quota = int(user.get("quota_bytes", 0))
    used = int(user.get("used_bytes", 0))
    remaining = max(quota - used, 0)
    expire_at = int(user.get("expire_at", 0))
    now = int(time.time())
    expire_in_s = max(expire_at - now, 0) if expire_at > 0 else 0
    expire_in_days = (expire_in_s + 86399) // 86400 if expire_in_s > 0 else 0

    created_at = user.get("created_at", "")
    expire_str = datetime.fromtimestamp(expire_at, TZ).strftime("%Y-%m-%d %H:%M:%S WIB") if expire_at > 0 else "-"
    created_str = created_at
    try:
        created_dt = datetime.strptime(created_at, "%Y-%m-%dT%H:%M:%SZ").replace(tzinfo=timezone.utc).astimezone(TZ)
        created_str = created_dt.strftime("%Y-%m-%d %H:%M:%S WIB")
    except Exception:
        pass

    return {
        "email": email,
        "enabled": bool(user.get("enabled", False)),
        "quota_bytes": quota,
        "used_bytes": used,
        "remaining_bytes": remaining,
        "quota_h": fmt_bytes(quota),
        "used_h": fmt_bytes(used),
        "remaining_h": fmt_bytes(remaining),
        "expire_at": expire_at,
        "expire_in_days": expire_in_days,
        "expire_wib": expire_str,
        "created_wib": created_str,
    }

# ====== Client config discovery (Clash & v2rayNG JSON) ======
def _read_text(path: str, max_bytes: int = 200_000) -> str:
    try:
        with open(path, "r", encoding="utf-8", errors="replace") as f:
            data = f.read(max_bytes + 1)
            if len(data) > max_bytes:
                data = data[:max_bytes] + "\n... (dipotong)"
            return data
    except Exception:
        return ""

def find_client_configs(email: str):
    """
    Clash TXT:   <uuid>-<email>.txt       (ambil terbaru 1)
    v2rayNG JSON:<uuid>-<email>-<TAG>.json (bisa banyak; urut terbaru dulu)
    Return: (clash_name, clash_text, v2_list)
            v2_list = [{name, text}]
    """
    clash_name = None
    clash_text = None
    v2_list = []

    # Clash TXT
    txt_candidates = sorted(
        glob.glob(os.path.join(CLIENT_DIR, f"*-{email}.txt")),
        key=os.path.getmtime,
        reverse=True,
    )
    if txt_candidates:
        clash_path = txt_candidates[0]
        clash_name = os.path.basename(clash_path)
        clash_text = _read_text(clash_path)

    # v2rayNG JSON
    json_candidates = sorted(
        glob.glob(os.path.join(CLIENT_DIR, f"*-{email}-*.json")),
        key=os.path.getmtime,
        reverse=True,
    )
    for p in json_candidates:
        v2_list.append({
            "name": os.path.basename(p),
            "text": _read_text(p),
        })

    return clash_name, clash_text, v2_list

# ====== HTML template ======
HTML_TPL = r"""
<!doctype html>
<html>
<head>
  <meta charset="utf-8"/><meta name="viewport" content="width=device-width, initial-scale=1"/>
  <title>Subscription · {{email}}</title>
  <style>
    :root{
      --bg:#0b1220;--card:#0f172a;--mut:#9aa4b2;--fg:#e7eaee;--key:#8fb7ff;--line:#2a3342;
      --chip:#1f2937;--btn:#1e293b;--btnh:#334155
    }
.footer a.link{ color:#a7f3d0; text-decoration:none; }
.footer a.link:hover{ text-decoration:underline; }
  /* ===== Base (mobile-first) ===== */
    html,body{
      background:var(--bg);color:var(--fg);
      font-family:system-ui,-apple-system,Segoe UI,Roboto,Inter,Arial,sans-serif;
      font-size:14.5px;   /* mobile base */
      line-height:1.4;
    }
    body{max-width:860px;margin:0 auto;padding:12px}
    .card{background:var(--card);border:1px solid var(--line);border-radius:14px;padding:12px 14px;margin:10px 0}
    h1{font-weight:600;font-size:clamp(17px,3.8vw,20px);margin:0 0 8px}
    .grid{display:grid;grid-template-columns:1fr;gap:8px}
    .row{display:flex;gap:8px;align-items:baseline}
    .k{min-width:92px;color:var(--key);font-size:clamp(12px,3.2vw,14px)}
    .v{font-size:clamp(12px,3.2vw,14px)}
    .muted{color:var(--mut);font-size:clamp(11px,2.9vw,12px)}
    .chips{display:flex;gap:6px;flex-wrap:wrap;margin-top:6px}
    .chip{font-size:12px;padding:4px 8px;border-radius:999px;background:var(--chip);border:1px solid var(--line)}
    .iplist{display:grid;grid-template-columns:1fr;gap:6px;margin-top:6px}
    .ip{font:12px ui-monospace,Menlo,Consolas,monospace;color:#c7f9e5;word-break:break-all}

    /* Accordion */
    details{border:1px solid var(--line);border-radius:12px;margin:10px 0;background:#0c1526}
    summary{list-style:none;cursor:pointer;padding:10px 12px;display:flex;justify-content:space-between;align-items:center;gap:8px}
    summary::-webkit-details-marker{display:none}
    .ttl{font-weight:600;font-size:clamp(13px,3.4vw,15px)}
    .actions{display:flex;gap:8px}
    .btn{appearance:none;border:0;border-radius:8px;background:var(--btn);padding:6px 10px;color:var(--fg);cursor:pointer;font-size:12px}
    .btn:hover{background:var(--btnh)}
    .dl{color:#a7f3d0;text-decoration:none}
    .box{border-top:1px dashed var(--line)}
    pre{margin:0;padding:10px 12px;white-space:pre-wrap;word-break:break-word;background:#0b1220}
    code{font:12px ui-monospace,Menlo,Consolas,monospace}

    .footer{margin:10px 0 4px;text-align:center}

    /* ===== Tablet (>=768px) — sedikit lebih besar ===== */
    @media (min-width:768px){
      html,body{font-size:15.5px}
      body{padding:16px}
      .card{padding:14px 16px}
      h1{font-size:clamp(18px,2.2vw,22px)}
      .k,.v{font-size:15px}
      .muted{font-size:13px}
      code{font-size:13px}
      .grid{grid-template-columns:1fr 1fr}
    }

    /* ===== Desktop (>=1024px) — nyaman dibaca monitor ===== */
    @media (min-width:1024px){
      html,body{font-size:16.5px}
      body{max-width:980px}
      .card{padding:16px 18px}
      h1{font-size:clamp(20px,1.8vw,24px)}
      .k,.v{font-size:16px}
      .muted{font-size:14px}
      code{font-size:14px}
    }
  </style>
</head>
<body>

  <!-- Kartu status -->
  <div class="card">
    <h1>{{email}}</h1>
    <div class="grid">
      <div class="row"><div class="k">Status</div><div class="v">{{ 'Aktif' if enabled else 'Nonaktif' }}</div></div>
      <div class="row"><div class="k">Kuota</div><div class="v">{{ used_h }} / {{ quota_h }} <span class="muted">· Sisa {{ remaining_h }}</span></div></div>
      <div class="row"><div class="k">Expired</div><div class="v">{{ expire_wib }} <span class="muted">(~{{ expire_in_days }} hari)</span></div></div>
      <div class="row"><div class="k">Dibuat</div><div class="v">{{ created_wib }}</div></div>
    </div>

    <div class="chips">
      <div class="chip">Window: {{window}}s</div>
      <div class="chip">Zona: WIB</div>
      <div class="chip">IP aktif: {{ ips|length }}</div>
    </div>

    <div class="iplist">
      {% if ips %}
        {% for ip, ts in ips %}<div class="ip">{{ip}}</div>{% endfor %}
      {% else %}
        <div class="muted">Tidak ada koneksi aktif dalam {{window}} detik terakhir.</div>
      {% endif %}
    </div>
  </div>

  {% if clash_text or v2_list %}
  <div class="card">
    <h1>Config Client</h1>

    {% if clash_text %}
    <details open>
      <summary>
        <span class="ttl">Clash (TXT{% if clash_name %}: {{clash_name}}{% endif %})</span>
        <span class="actions">
          <button class="btn" onclick="copyBox('clash_box')">Copy</button>
          {% if clash_name %}<a class="btn dl" href="/{{clash_name}}" download>Download</a>{% endif %}
        </span>
      </summary>
      <div class="box">
        <pre id="clash_box"><code>{{ clash_text }}</code></pre>
      </div>
    </details>
    {% endif %}

    {% if v2_list %}
      {% for item in v2_list %}
      <details>
        <summary>
          <span class="ttl">v2rayNG JSON — {{ item.name }}</span>
          <span class="actions">
            <button class="btn" onclick="copyBox('v2_{{ loop.index }}')">Copy</button>
            <a class="btn dl" href="/{{item.name}}" download>Download</a>
          </span>
        </summary>
        <div class="box">
          <pre id="v2_{{ loop.index }}"><code>{{ item.text }}</code></pre>
        </div>
      </details>
      {% endfor %}
    {% endif %}
  </div>
  {% endif %}

<div class="footer muted">
  © <a href="https://t.me/LingVPN" class="link" target="_blank" rel="noopener">LingVPN</a> · Fast & Reliable
</div>

  <script>
    async function copyBox(id){
      const el = document.getElementById(id);
      if(!el) return;
      const txt = el.innerText;
      try{
        await navigator.clipboard.writeText(txt);
        toast('Copied');
      }catch(e){
        const r = document.createRange(); r.selectNode(el);
        const s = getSelection(); s.removeAllRanges(); s.addRange(r);
        document.execCommand('copy'); s.removeAllRanges();
        toast('Copied');
      }
    }
    function toast(t){
      try{
        let x = document.getElementById('toast');
        if(!x){
          x = document.createElement('div'); x.id='toast';
          x.style.position='fixed'; x.style.left='50%'; x.style.bottom='18px';
          x.style.transform='translateX(-50%)'; x.style.background='#111827';
          x.style.border='1px solid #374151'; x.style.color='#e5e7eb';
          x.style.padding='8px 12px'; x.style.borderRadius='10px'; x.style.fontSize='12px';
          x.style.zIndex='9999'; document.body.appendChild(x);
        }
        x.textContent=t; x.style.opacity='1';
        setTimeout(()=>{ x.style.transition='opacity .4s'; x.style.opacity='0'; }, 900);
      }catch(_){}
    }
  </script>
</body>
</html>
"""

# ========= ROUTES =========
@app.get(f"/{BASE_JSON}/<email>")
def api(email):
    token = request.args.get("token", "")
    if not verify(email, token):
        abort(403)
    info = user_view(email)
    if not info:
        abort(404)
    ips = extract_ips_from_log(email, WINDOW_SEC)
    info["online_ips"] = sorted(ips.keys())
    info["online_ip_count"] = len(info["online_ips"])
    info["window_sec"] = WINDOW_SEC
    return jsonify(info)

@app.get(f"/{BASE_HTML}/<email>")
def page(email):
    token = request.args.get("token", "")
    if not verify(email, token):
        abort(403)
    info = user_view(email)
    if not info:
        abort(404)

    ips = extract_ips_from_log(email, WINDOW_SEC)
    ips_sorted = sorted(ips.items(), key=lambda kv: kv[1], reverse=True)

    clash_name, clash_text, v2_list = find_client_configs(email)

    return render_template_string(
        HTML_TPL,
        email=email,
        enabled=info["enabled"],
        used_h=info["used_h"],
        quota_h=info["quota_h"],
        remaining_h=info["remaining_h"],
        expire_wib=info["expire_wib"],
        expire_in_days=info["expire_in_days"],
        created_wib=info["created_wib"],
        ips=ips_sorted,
        window=WINDOW_SEC,
        clash_name=clash_name,
        clash_text=clash_text,
        v2_list=v2_list,
    )

@app.get("/")
def root():
    return jsonify({
        "ok": True,
        "usage": f"/{BASE_HTML}/<email>?token=...  or  /{BASE_JSON}/<email>?token=...",
        "window_sec": WINDOW_SEC,
        "tz_offset_hours": int(TZ.utcoffset(datetime.now()).total_seconds()/3600),
        "base_html": BASE_HTML,
        "base_json": BASE_JSON,
        "client_dir": CLIENT_DIR,
    })

if __name__ == "__main__":
    app.run(host="127.0.0.1", port=int(os.environ.get("PORT","8989")))
